/*******************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
** This file is part of the OfficeViewer project.
**
** Redistribution and use in source and binary forms,
** with or without modification, are permitted provided
** that the following conditions are met:
**
** * Redistributions of source code must retain the above copyright notice,
**   this list of conditions and the following disclaimer.
** * Redistributions in binary form must reproduce the above copyright notice,
**   this list of conditions and the following disclaimer
**   in the documentation and/or other materials provided with the distribution.
** * Neither the name of the copyright holder nor the names of its contributors
**   may be used to endorse or promote products derived from this software
**   without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
** THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
** FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
** EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*******************************************************************************/


import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.docviewer 1.0
import "utils.js" as Utils

Page {
    id: page

    property url source
    property int indexCount
    property string mimeType

    FileInfo {
        id: info
        source: page.source
    }

    SilicaFlickable {
        id: flickable
        anchors.fill: parent
        contentHeight: contentColumn.height + Theme.paddingLarge

        Column {
            id: contentColumn
            width: parent.width

            PageHeader {
                id: detailsHeader
                //: Details page title
                //% "Details"
                title: qsTrId("docviewer-he-details")
            }

            DetailItem {
                //: File path detail of the document
                //% "File path"
                label: qsTrId("docviewer-la-filepath")
                value: info.filePath
                alignment: Qt.AlignLeft
            }

            DetailItem {
                //: File size detail of the document
                //% "Size"
                label: qsTrId("docviewer-la-filesize")
                value: Format.formatFileSize(info.fileSize)
                alignment: Qt.AlignLeft
            }

            DetailItem {
                //: File type detail of the document
                //% "Type"
                label: qsTrId("docviewer-la-filetype")
                value: info.mimeTypeComment
                alignment: Qt.AlignLeft
            }

            DetailItem {
                //: Last modified date of the document
                //% "Last modified"
                label: qsTrId("docviewer-la-lastmodified")
                value: Format.formatDate(info.modifiedDate, Format.DateFull)
                alignment: Qt.AlignLeft
            }

            DetailItem {
                label: {
                    switch (Utils.docviewerType(page.mimeType)) {
                        case "calc":
                            //: Sheet count of the spreadsheet
                            //% "Sheets"
                            return qsTrId("docviewer-la-sheetcount")
                        case "impress":
                            //: Slide count detail of the presentation
                            //% "Slides"
                            return qsTrId("docviewer-la-slidecount")
                        case "writer":
                            //: Page count of the text document
                            //% "Page count"
                            return qsTrId("docviewer-la-pagecount")
                        case "pdf":
                            //: Page count of the text document
                            //% "Page count"
                            return qsTrId("docviewer-la-pagecount")
                        default:
                            //: Index count for unknown document types.
                            //% "Index count"
                            return qsTrId("docviewer-la-indexcount")
                    }
                }
                value: page.indexCount
                alignment: Qt.AlignLeft
            }
        }

        VerticalScrollDecorator {}
    }
}
