/*
 * Copyright (c) 2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#pragma once

#include <QQuickItem>
#include <QImage>
#include <QPointer>

class BasePage;
class QTimer;
class QSGTexture;
class QPropertyAnimation;
class BitmapLoaderWorker;
class PageTile : public QQuickItem
{
    Q_OBJECT

    Q_PROPERTY(bool renderable READ renderable WRITE setRenderable NOTIFY renderableChanged)

public:
    explicit PageTile(QQuickItem *parent = nullptr);
     ~PageTile() override;

    QSGNode *updatePaintNode(QSGNode *oldNode, UpdatePaintNodeData *) override;
    void geometryChanged(const QRectF &newGeometry, const QRectF &oldGeometry) override;

    void setPageSource(QSharedPointer<BasePage> pageSource);
    bool renderable() const;
    bool isBitmap() const;
    void setDebugText(const QString &newDebugText);

public slots:
    void setImageScale(qreal imageScale);
    void render(bool force = false);
    void setRenderable(bool renderable);

signals:
    void bitmapError();
    void partReady();
    void animationEnded();
    void renderableChanged(bool renderable);
    void stopRender();

private slots:
    void _loadBitmap();
    void _clearImage();
    void _debugDraw(QImage *image);
    void _debugDraw(QImage *image, QColor color);
    void _updateDebugStatus();

private:
    QImage m_pagePart;
    QSharedPointer<BasePage> m_pageSource;
    bool m_needUpdateImage;
    bool m_needClearImage;
    bool m_renderInProcess;
    bool m_renderable;
    bool m_forceRender;
    QTimer *m_timer;
    QTimer *m_timerClear;
    qreal m_imageScale;
    QPropertyAnimation *m_animation;
    QString m_debugText;
    bool m_debugCancel;
    bool m_debugError;
    QPointer<BitmapLoaderWorker> m_loader;
};
