/*
 * Copyright (c) 2022 - 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#pragma once

#include <QQuickItem>

#include "docviewtypes.h"

class BasePage;
class BaseAnnotation;
class DocumentMapper;
class QSGTexture;
class MultiPageTile;
class BaseDocItem;
class DocItemAnnotation;
class DocItemComment;
class DocItemSearchHighlight;
class BackgroundPage;
class PageContainer : public QQuickItem
{
    Q_OBJECT

    Q_PROPERTY(QSizeF requestedSize READ requestedSize WRITE setRequestedSize NOTIFY requestedSizeChanged)
    Q_PROPERTY(Qt::Orientation orientation READ orientation WRITE setOrientation NOTIFY orientationChanged)
    Q_PROPERTY(qreal scale READ scale WRITE setScale NOTIFY scaleChanged)
    Q_PROPERTY(QColor highLightColor READ highLightColor WRITE setHighLightColor NOTIFY highLightColorChanged)
    Q_PROPERTY(bool annotationsPaint READ annotationsPaint WRITE setAnnotationsPaint NOTIFY annotationsPaintChanged)
    Q_PROPERTY(bool notesPaint READ notesPaint WRITE setNotesPaint NOTIFY notesPaintChanged)

public:
    explicit PageContainer(QQuickItem *parent = nullptr);
    ~PageContainer() override;

    QSGNode *updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *) override;
    void geometryChanged(const QRectF &newGeometry, const QRectF &oldGeometry) override;

    QSizeF requestedSize() const;
    Qt::Orientation orientation() const;
    qreal scale() const;
    void setPageSource(QSharedPointer<BasePage> pageSource);
    void setPageGeometry(const PageGeometry &pg);
    int index() const;
    void setMapper(DocumentMapper *mapper);
    static QSizeF pageCurrentSize(const PageGeometry &pageGeometry, const QSizeF &requestedSize, Qt::Orientation orientation, qreal scale);
    void setVisibleArea(const QRectF &visibleArea);
    bool annotationsPaint() const;
    bool notesPaint() const;
    QSharedPointer<BasePage> source() const;
    void addAnnotation(const QRectF &rect, const QColor &color,
                       const QString &author, const QString &content);
    qreal imageScale() const;

    const QColor &highLightColor() const;
    void setHighLightColor(const QColor &newHighLightColor);
    qreal pageRate() const;

public slots:
    void setRequestedSize(QSizeF requestedSize);
    void setOrientation(Qt::Orientation orientation);
    void setScale(qreal scale);
    void setAnnotationsPaint(bool annotationsPaint);
    void setNotesPaint(bool notesPaint);
    void forceUpdateRect(const QRectF &visibleArea);
    void setColorSelected(const QColor &color);

signals:
    void requestedSizeChanged(QSizeF requestedSize);
    void orientationChanged(Qt::Orientation orientation);
    void scaleChanged(qreal scale);
    void pageReady();
    void pageChanged();
    void annotationsPaintChanged(bool annotationsPaint);
    void annotationsLoaded();
    void annotationActivate(BaseAnnotation *);
    void notesPaintChanged(bool notesPaint);
    void noteActivate(BaseAnnotation *);
    void cellActivate(BaseAnnotation *);
    void highLightColorChanged();

private slots:
    void _correctSize();
    void _tailorise();
    void _updateVisible();
    void _loadAnnotations();
    void _prepareBackgroundPage();

private:
    PageGeometry m_pageGeometry;
    QSizeF m_requestedSize;
    QSizeF m_pageImageSize;
    Qt::Orientation m_orientation;
    QSharedPointer<BasePage> m_pageSource;
    qreal m_scale;
    qreal m_imageScale;
    DocumentMapper *m_mapper;
    QHash<int, MultiPageTile *> m_tilesMap;
    BackgroundPage *m_backgroundPage;
    qreal m_maxTileZ;
    QRectF m_visibleArea;
    quint16 m_tileSize;
    bool m_annotationsPaint;
    bool m_notesPaint;
    bool m_cellSelectedPaint;
    bool m_allTilesReady;
    QList<BaseDocItem *> m_annotationsItems;
    QList<BaseDocItem *> m_commentsItems;
    QList<BaseDocItem *> m_highlightedItems;
    QList<BaseDocItem *> m_currentCellItems;
    QColor m_highLightColor;
    QQuickItem *m_cursor;
};
