/*
 * Copyright (c) 2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QMutexLocker>
#include <QThread>
#include <QTimer>

#include "debug.h"
#include "basepage.h"

#include "bitmaploadworker.h"

QMutex BitmapLoaderWorker::m_mutex;

BitmapLoaderWorker::BitmapLoaderWorker(QSharedPointer<BasePage> p, qreal scaleX, qreal scaleY, int flags, qreal zoom , QPointF bias, qreal pageScale):
    m_page(p),
    m_scaleX(scaleX),
    m_scaleY(scaleY),
    m_flags(flags),
    m_zoom(zoom),
    m_bias(bias),
    m_pageScale(pageScale),
    m_isCanceled(false),
    m_loaded(false)
{
}

BitmapLoaderWorker::~BitmapLoaderWorker()
{
    cancel();
    qCDebug(lcDocviewerRender) << "Destroy loader page index:" << m_page->pageIndex();
#ifdef DEBUG_MODE
    m_page->stopBitmapPart(m_scaleX, m_scaleY, m_zoom, m_bias);
#endif
}

void BitmapLoaderWorker::run() {
    if (!m_page) {
        emit done({  });
        return;
    }

    if (m_isCanceled) {
        emit done({  });
        return;
    }

    setAutoDelete(false);
    connect(&m_watcher, &QFutureWatcher<QImage>::finished, this, &BitmapLoaderWorker::_getResult);
    QMutexLocker lock(&m_mutex);
    m_watcher.setFuture(m_page->bitmapPart(m_scaleX, m_scaleY, m_flags, m_zoom, m_bias, m_pageScale));

    // If after 30 seconds no data has been received, then delete the loader
    QTimer::singleShot(30 * 1000, this, &BitmapLoaderWorker::_getResult);
}

void BitmapLoaderWorker::cancel()
{
    if (m_watcher.isRunning())
        m_watcher.cancel();

    if (m_isCanceled)
        return;

    if (m_page && !m_loaded)
        m_page->stopBitmapPart(m_scaleX, m_scaleY, m_zoom, m_bias);
    else
        qCDebug(lcDocviewerRender) << "Not cancel";

    m_isCanceled = true;
    setAutoDelete(true);
    deleteLater();
}

void BitmapLoaderWorker::_getResult()
{

    if (m_watcher.isFinished() && !m_watcher.isCanceled())
        emit done(m_watcher.result());
    else
        emit done({  });

    m_loaded = true;
    setAutoDelete(true);
    deleteLater();
}
