/*
 * Copyright (c) 2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QQuickWindow>
#include <QSGSimpleTextureNode>

#include "basedocitem.h"

BaseDocItem::BaseDocItem(const QColor &color, const QColor &highlightcolor, QQuickItem *parent, BaseAnnotation *source)
    : QQuickItem(parent)
    , m_needUpdateImage(false)
    , m_source(source)
    , m_highlighted(false)
    , m_color(color)
    , m_highlightcolor(highlightcolor)
{
    setFlag(QQuickItem::ItemHasContents, true);
    setAcceptedMouseButtons(Qt::AllButtons);

    connect(this, &BaseDocItem::xChanged, this, &BaseDocItem::update);
    connect(this, &BaseDocItem::yChanged, this, &BaseDocItem::update);
    connect(this, &BaseDocItem::widthChanged, this, &BaseDocItem::update);
    connect(this, &BaseDocItem::heightChanged, this, &BaseDocItem::update);

    connect(parent, &QQuickItem::yChanged, this, &BaseDocItem::clearHighlight);
    connect(parent, &QQuickItem::xChanged, this, &BaseDocItem::clearHighlight);

    m_paper = QImage(1, 1, QImage::Format_RGBA8888);

    setColor(color);
}

BaseAnnotation *BaseDocItem::source() const
{
    return m_source;
}

QSGNode *BaseDocItem::updatePaintNode(QSGNode *oldNode, UpdatePaintNodeData *)
{
    if (height() <= 0 || width() <= 0)
        return nullptr;

    auto node = static_cast<QSGSimpleTextureNode *>(oldNode);

    if (node == nullptr) {
        node = new QSGSimpleTextureNode();
        node->setOwnsTexture(true);
    }

    if (m_needUpdateImage || node->texture() == nullptr) {
        if (node->texture() != nullptr)
            node->texture()->deleteLater();

        node->setTexture(window()->createTextureFromImage(m_paper));

        m_needUpdateImage = false;
    }

    node->setRect(boundingRect());

    return node;
}

bool BaseDocItem::event(QEvent *event)
{
    if (opacity() < 1.0f)
        return QQuickItem::event(event);

    switch (event->type()) {
    case QEvent::MouseButtonPress:
        setHighlight(true);
        return true;

    case QEvent::MouseButtonRelease:
        if (source() != nullptr)
            emit triggered(source());
        setHighlight(false);
        break;

    case QEvent::MouseMove:
        setHighlight(false);
        break;

    default:
        break;
    }

    return QQuickItem::event(event);
}

void BaseDocItem::clearHighlight()
{
    setHighlight(false);
}

void BaseDocItem::setHighlight(bool highlight)
{
    if (m_highlighted != highlight) {
        m_highlighted = highlight;
        m_paper.fill(m_highlighted ? m_highlightcolor : m_color);
        m_needUpdateImage = true;
        update();
    }
}

void BaseDocItem::setColor(const QColor &newColor)
{
    m_color = newColor;
    if (!m_highlighted) {
        m_paper.fill(m_color);
        m_needUpdateImage = true;
        update();
    }
}

void BaseDocItem::setHighlightcolor(const QColor &newHighlightcolor)
{
    m_highlightcolor = newHighlightcolor;
    if (!m_highlighted) {
        m_paper.fill(m_highlightcolor);
        m_needUpdateImage = true;
        update();
    }
}

bool BaseDocItem::highlighted() const
{
    return m_highlighted;
}

const QColor &BaseDocItem::highlightcolor() const
{
    return m_highlightcolor;
}

const QColor &BaseDocItem::color() const
{
    return m_color;
}
