/*
 * Copyright (C) 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

import QtQuick 2.6
import Sailfish.Silica 1.0

Row {
    id: root

    property bool selectAllEnabled
    property bool portrait: true

    property real _buttonWidth: width / (repeater.count + 1)

    readonly property var defaultButtons: {
        var buttons = []

        buttons.push({
            "icon": "image://theme/icon-splus-share",
            //% "Share"
            "label": qsTrId("docviewer-la-share"),
            "action": function () {
                clear()
                share()
            }
        })

        buttons.push({
            "icon": "image://theme/icon-splus-clipboard",
            //% "Copy"
            "label": qsTrId("docviewer-la-copy"),
            "action": function () {
                clear()
                copy()
            }
        })

        buttons.push({
            "icon": "image://theme/icon-m-input-remove",
            //% "Clear"
            "label": qsTrId("docviewer-la-clear"),
            "action": clear
        })

        return buttons
    }

    property var buttons: defaultButtons

    property int fontSize: {
        var size = Theme.fontSizeTiny

        if (!portrait && size > Theme.fontSizeTinyBase) {
            var string = ""
            for (var i = 0; i < buttons.length; ++i) {
                string = string + (buttons[i].label || "")
            }

            var requiredWidth = metrics.advanceWidth(string) + (buttons.length * (Theme.itemSizeSmall + (2 * horizontalSpacing)))

            if (requiredWidth >= width - leftPadding - rightPadding) {
                size = Theme.fontSizeTinyBase
            }
        }
        return size
    }
    property real horizontalSpacing

    signal share
    signal copy
    signal clear

    height: Theme.itemSizeSmall + Theme.paddingLarge

    FontMetrics {
        id: metrics
        font.pixelSize: Theme.fontSizeTiny
    }

    Repeater {
        id: repeater

        model: root.buttons
        delegate: root.portrait ? portraitButtonDelegate : landscapeButtonDelegate
    }

    Component {
        id: portraitButtonDelegate

        IconButton {
            id: portraitButton

            width: (root.width - root.leftPadding - root.rightPadding) / Math.max(1, repeater.count)
            height: root.height
            icon.source: modelData.icon
            icon.sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)
            icon.anchors.verticalCenterOffset: -(portraitLabel.baselineOffset - Theme.paddingSmall)
            onClicked: root.buttons[index].action()

            Label {
                id: portraitLabel

                x: (portraitButton.width - width) / 2
                y: portraitButton.icon.y + portraitButton.icon.height - Theme.paddingSmall

                text: modelData.label
                font.pixelSize: root.fontSize
            }
        }
    }
    Component {
        id: landscapeButtonDelegate

        IconButton {
            id: landscapeButton

            width: (root.width - root.leftPadding - root.rightPadding) / Math.max(1, repeater.count)
            height: root.height
            icon.source: modelData.icon
            icon.anchors.horizontalCenterOffset: -(Math.min(landscapeLabel.width, landscapeLabel.implicitWidth) + root.horizontalSpacing) / 2

            onClicked: root.buttons[index].action()

            Label {
                id: landscapeLabel
                x: landscapeButton.icon.x + landscapeButton.icon.width + root.horizontalSpacing
                y: (landscapeButton.height - height) / 2
                width: landscapeButton.width - landscapeButton.icon.width - (2 * root.horizontalSpacing)

                text: modelData.label
                font.pixelSize: root.fontSize
                truncationMode: TruncationMode.Fade
            }
        }
    }
}
