/*
 * Copyright (C) 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

import QtQuick 2.1
import Sailfish.Silica 1.0
import ".."

Item {
    id: handle

    property real fixedX
    property real fixedY
    property real scale
    property real minimumX
    property real maximumX

    property QtObject contentItem

    property bool moving
    readonly property bool dragActive: mouseArea.drag.active

    property string markerTag
    // Mask area for touch handling, should cover whole mousearea
    property rect mask: Qt.rect(0, 0, 0, 0)

    property bool _startMarker: markerTag === "start"

    width: Theme.itemSizeExtraSmall
    height: width

    onFixedXChanged: {
        if (!moving)
            timer.restart()

        _startMarker ? handle.x = fixedX - width : handle.x = fixedX
    }
    onFixedYChanged: {
        if (!moving)
            timer.restart()

        handle.y = fixedY
    }

    onDragActiveChanged: {
        if (dragActive) {
            moving = true
        } else {
            moving = false
            timer.restart()
        }
    }

    onVisibleChanged: {
        if (visible) {
            showAnimation.restart()
        } else {
            mask = Qt.rect(0, 0, 0, 0)
        }
    }

    Image {
        x: _startMarker ? handle.width - width : 0
        width: Theme.iconSizeSmallPlus
        height: width
        transformOrigin: _startMarker ? Item.TopRight : Item.TopLeft
        source: "image://theme/icon-m-textselection-" + markerTag
    }

    MouseArea {
        id: mouseArea

        property real previousX: -1
        property real previousY: -1

        anchors {
            fill: parent
            leftMargin: -Theme.paddingLarge
            rightMargin: -Theme.paddingLarge
            bottomMargin: -Theme.paddingLarge
        }

        drag {
            target: parent
            axis: Drag.XandYAxis
            minimumX: handle.minimumX - width + Theme.iconSizeSmallPlus
            minimumY: 0
            maximumX: handle.maximumX
            maximumY: contentItem.contentHeight
        }

        preventStealing: true

        onPositionChanged: {
            var targetX = mouse.x
            var targetY = mouse.y

            if (!handle.moving || Math.abs(targetX - previousX) < 0.99 && Math.abs(targetY - previousY) < 0.99) {
                return
            }

            if (_startMarker)
                contentItem.startSelectText(Qt.point(handle.x + handle.width, handle.y))
            else
                contentItem.endSelectText(Qt.point(handle.x, handle.y))

            previousX = targetX
            previousY = targetY
        }
    }

    NumberAnimation {
        id: showAnimation

        target: handle
        property: "opacity"
        from: 0
        to: 1.0
    }

    ParallelAnimation {
        id: targetPositionAnimation

        NumberAnimation {
            target: handle
            property: "x"
            to: _startMarker ? fixedX - width : fixedX
            duration: 100
            easing.type: Easing.InOutQuad
        }
        NumberAnimation {
            target: handle
            property: "y"
            to: handle.fixedY
            duration: 100
            easing.type: Easing.InOutQuad
        }
    }

    Timer {
        id: timer

        onTriggered: targetPositionAnimation.start()
        interval: 300
    }
}
