/*
 * Copyright (c) 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Nemo.FileManager 1.0
import "utils.js" as Utils

Dialog {
    id: root

    property string path
    property string baseName
    property string sourceMimeType
    property alias fileExtension: comboBox.value

    canAccept: !fileName.errorHighlight

    DialogHeader {
        id: dialogHeader

        //% "Save"
        acceptText: qsTrId("docviewer-he-save")
        //% "Cancel"
        cancelText: qsTrId("docviewer-he-cancel")
    }

    SilicaFlickable {
        anchors {
            top: dialogHeader.bottom
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }

        contentHeight: content.height

        Column {
            id: content

            width: parent.width
            spacing: Theme.paddingLarge

            InfoLabel {
                horizontalAlignment: Text.AlignLeft
                text: qsTrId("docviewer-ti-save_as")
            }

            TextField {
                id: fileName

                property bool fileExist
                readonly property var regExp: new RegExp("[\/*\?<>\|]+")

                function checkFileName() {
                    fileExist = FileEngine.exists("%1/%2%3".arg(path).arg(text).arg(comboBox.value))
                    errorHighlight = (regExp.test(text) || fileExist)
                }

                width: parent.width
                font.pixelSize: Theme.fontSizeMedium
                label: errorHighlight
                       ? (fileExist
                          //% "The same name already exists"
                          ? qsTrId("docviewer-la-name_already_exist")
                            //% "Invalid file name"
                          : qsTrId("docviewer-la-invalid_filename"))
                         //% "Title"
                       : qsTrId("docviewer-la-title_save_as")

                text: baseName
                placeholderText: qsTrId("docviewer-la-save_as")
                onFocusChanged: if (focus) selectAll()
                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: accept()
                onTextChanged: {
                    checkFileName()
                    root.baseName = text
                }
                Component.onCompleted: {
                    text = baseName
                    checkFileName()
                }
            }

            ComboBox {
                id: comboBox
                property var types: {
                    switch(Utils.docviewerType(sourceMimeType)) {
                    case "calc":
                        return [".ods", ".xls", ".xlsx"]
                    case "impress":
                        return [".odp", ".ppt", ".pptx"]
                    case "writer":
                        return [".odt", ".doc", ".docx"]
                    }
                }

                //% "Format file"
                label: qsTrId("docviewer-la-format_file")

                menu: ContextMenu {
                    Repeater {
                        model: comboBox.types
                        MenuItem {
                            text: modelData
                        }
                    }
                }

                onValueChanged: fileName.checkFileName()
            }
        }
    }
}
