/**
 * SPDX-FileCopyrightText: Copyright 2020 - 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: GPLv2
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import ru.omp.documentprovider.libreoffice 1.0
import ru.omp.docviewer 1.0
import "components"

Rectangle {
    id: root

    property Component header
    property alias count: docView.count
    property alias documentProvider: docView.documentProvider
    property alias currentIndex: docView.currentIndex
    property alias catchBound: docView.catchBound
    property alias typeScroll: docView.typeScroll
    property alias annotationsPaint: docView.annotationsPaint
    property alias notesPaint: docView.notesPaint
    property alias zoom: docView.itemScale
    property alias contentTopMargin: docView.contentTopMargin
    property alias path: docView.path
    property alias status: docView.status
    property alias bookmarksModel: docView.bookmarksModel
    property alias fileVersion: docView.fileVersion
    property alias documentEdited: docView.documentEdited
    property alias contentX: viewFlick.contentX
    property alias contentY: viewFlick.contentY
    property alias contentWidth: viewFlick.contentWidth
    property alias contentHeight: viewFlick.contentHeight
    property alias moving: viewFlick.moving
    property alias verticalVelocity: viewFlick.verticalVelocity
    property alias topMargin: docView.topMargin
    property alias selectionMode: docView.selectionMode
    property alias selectionText: docView.selectionText
    property var orientation

    signal clicked()
    signal clickedUrl(string url)
    signal clickedCell(string cellContent)
    signal noteActivate(string text, string author, string dateTime)
    signal contentChanged()
    signal documentSaved(bool saveStatus)
    signal holding(var lastMouse, var screenCoordinates)

    function cancelFlick() {
        viewFlick.cancelFlick()
    }

    function correctPosition() {
        if (orientationChanging.running) {
            if (docView.oldContentWidth === 0)
                docView.oldContentWidth = docView.contentWidth !== 0 ? docView.contentWidth : 1
            var rX = docView.oldContentX / docView.oldContentWidth
            docView.contentX = rX * docView.contentWidth
            if (docView.oldContentHeight === 0)
                docView.oldContentHeight = docView.contentHeight != 0 ? docView.contentHeight : 1
            var rY = (docView.oldContentY - docView.oldTopMargin) / docView.oldContentHeight
            docView.contentY = rY * docView.contentHeight + docView.topMargin
        }

        viewFlick.contentY = docView.contentY
        viewFlick.contentX = docView.contentX

        viewFlick.returnToBounds()
    }
    function goToPage(pageIndex) {
        viewPositionBlocker.start()
        docView.positionViewAtIndex(pageIndex)
    }
    function saveDocumentAs(path, format) {
        console.log(path, format)
        docView.saveDocumentAs(path, format)
    }
    function addAnnotation(rect, author, content, color) {
        rect.x += viewFlick.contentX
        rect.y += viewFlick.contentY
        docView.addAnnotation(rect, color, author, content)
    }
    function startSearch(searchString, backward) {
        docView.startSearch(searchString, backward ? LoDocument.SearchBackward : LoDocument.SearchForward)
    }
    function stopSearch() {
        docView.stopSearch()
    }
    function resetSelection() {
        docView.resetSelectText()
    }

    function updateOrientation() {
        docView.oldContentHeight = docView.contentHeight
        docView.oldContentWidth = docView.contentWidth
        docView.oldTopMargin = docView.topMargin
        docView.oldContentY = docView.contentY
        docView.oldContentX = docView.contentX
    }

    onOrientationChanged: {
        orientationChanging.start()
        updateOrientation()
    }

    Timer {
        id: orientationChanging
        interval: 1000
    }

    color: "transparent"
    data: [
        DocViewPrivate {
            id: docView

            property int oldContentHeight
            property int oldContentWidth
            property int oldTopMargin
            property real oldContentY
            property real oldContentX

            parent: viewFlick
            height: parent.height
            width: parent.width
            anchors.top: parent.top
            highLightColor: Theme.secondaryHighlightColor
            cursorDelegate: Cursor {}
            startSelectionMarker: TextSelectionHandle {
                markerTag: "start"
                contentItem: docView
            }
            endSelectionMarker: TextSelectionHandle {
                markerTag: "end"
                contentItem: docView
            }
            onWidthChanged: {
                viewPositionBlocker.start()
                root.correctPosition()
            }
            onHeightChanged: {
                viewPositionBlocker.start()
                root.correctPosition()
            }
            onContentChanged: {
                root.correctPosition()
                root.contentChanged()
            }
            onTypeScrollChanged: root.correctPosition()
            onItemScaleChanged: root.correctPosition()
            onClickedUrl: {
                if (holdTimer.running) {
                    holdTimer.stop()
                    doubleClickTimer.stop()
                    root.clickedUrl(url)
                }
            }
            onCellClicked: {
                root.clickedCell(cellContent)
            }
            onClickedGoToPage: root.correctPosition()
            onNoteActivated: root.noteActivate(text, author, dateTime)
            onDocumentSaved: root.documentSaved(saveStatus)

            Rectangle {
                visible: count > 0 && zoom >= 1.0
                color: orientation === Qt.Vertical ? "white" : "transparent"
                width: parent.width
                height: topMargin
            }
        },
        SilicaFlickable {
            id: viewFlick

            property var fixedScaleValues: [0.5, 1.0, 1.5, 2]
            property real maximumItemScale: fixedScaleValues[fixedScaleValues.length - 1]
            property real minimumItemScale: fixedScaleValues[0]
            property real maximumPinchScale: 6.0

            property real oldContentY

            function scaleAroundPoint(center, newScale) {
                docView.scaleAroundPoint(center, newScale)
                root.correctPosition()
            }
            function clicked(mouse) {
                var screenX = mouse.x - contentX
                var screenY = mouse.y - contentY

                root.clicked()
            }
            function doubleClicked(mouse) {}

            contentHeight: Math.max(height, (docView.typeScroll === Qt.Vertical
                                    ? docView.contentHeight * docView.itemScale + docView.lastPageIndent
                                    : docView.contentHeight))

            contentWidth: Math.max(width, (docView.typeScroll === Qt.Vertical
                                   ? docView.contentWidth
                                   : docView.contentWidth * docView.itemScale + docView.lastPageIndent))
            flickableDirection: Flickable.AutoFlickDirection
            anchors.fill: parent

            onContentYChanged: {
                if (viewPositionBlocker.running)
                    return

                docView.contentY = contentY
            }
            onContentXChanged: {
                if (viewPositionBlocker.running)
                    return

                docView.contentX = contentX
            }

            Timer {
                id: viewPositionBlocker

                interval: 100
                repeat: false

                onTriggered: root.correctPosition()
            }

            ScrollDecorator {  }

            PinchArea {
                id: pinchArea

                property real startScale
                property bool moving: false

                anchors.fill: parent
                enabled: docView.status === LoDocument.Ready

                onMovingChanged: if (moving) root.moving()
                onPinchStarted: {
                    if (viewFlick.moving)
                        viewFlick.cancelFlick()

                    startScale = docView.itemScale
                    viewFlick.interactive = false
                }
                onPinchUpdated: {
                    viewFlick.scaleAroundPoint(pinch.center,
                                               Math.max(viewFlick.minimumItemScale,
                                                        Math.min(viewFlick.maximumPinchScale, startScale * pinch.scale)))
                }
                onPinchFinished: {
                    viewFlick.returnToBounds()
                    viewFlick.interactive = true
                }

                MouseArea {
                    id: mouseArea

                    anchors.fill: parent
                    enabled: parent.enabled
                    property point lastMouse
                    property double deltaX: 10.0
                    property double deltaY: 10.0
                    property int doubleClickLatency: 350
                    property int holdLatency: 800

                    signal click(var lastMouse)
                    signal doubleClick(var lastMouse)
                    signal hold(var lastMouse, var screenCoordinates)

                    onClick: viewFlick.clicked(lastMouse)
                    onDoubleClick: viewFlick.doubleClicked(lastMouse)
                    onHold: {
                        root.holding(lastMouse, screenCoordinates)
                        docView.mouseDoubleClick(Qt.point(lastMouse.x, lastMouse.y))
                    }
                    onPressed: {
                        lastMouse = Qt.point(mouseX, mouseY)
                        holdTimer.start()

                        docView.mousePress(Qt.point(mouseX, mouseY))
                    }
                    onReleased: {
                        lastMouse = Qt.point(mouseX, mouseY)

                        if (doubleClickTimer.running) {
                            doubleClickTimer.stop()
                            doubleClick(lastMouse)

                            if (holdTimer.running)
                                holdTimer.stop()

                            return
                        }

                        if (holdTimer.running) {
                            holdTimer.stop()
                            doubleClickTimer.start()
                        }

                        docView.mouseRelease(Qt.point(mouseX, mouseY))
                    }
                    onMouseXChanged: {
                        if (!holdTimer.running)
                            return

                        if (Math.abs(mouseX - lastMouse.x) > deltaX)
                            holdTimer.stop()
                    }
                    onMouseYChanged: {
                        if (!holdTimer.running)
                            return

                        if (Math.abs(mouseY - lastMouse.y) > deltaY)
                            holdTimer.stop()
                    }

                    Timer {
                        id: doubleClickTimer

                        interval: mouseArea.doubleClickLatency

                        onTriggered: mouseArea.click(mouseArea.lastMouse)
                    }
                    Timer {
                        id: holdTimer

                        interval: mouseArea.holdLatency
                        repeat: false

                        onTriggered: {
                            if (doubleClickTimer.running)
                                doubleClickTimer.stop()

                            mouseArea.hold(mouseArea.lastMouse,
                                           Qt.point(mouseArea.lastMouse.x - viewFlick.contentX,
                                                    mouseArea.lastMouse.y - viewFlick.contentY))
                        }
                    }
                }
            }
        }
    ]
}
