/*
 * Copyright (c) 2020 - 2022 Open Mobile Platform LLC.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

import QtQuick 2.0
import Sailfish.Silica 1.0

Item {
    id: root

    property var document
    property color color: Theme.primaryColor
    property alias backgroundColor: background.color
    property alias backgroundVisible: background.visible
    property bool opened: true
    property bool searchMode
    property bool allowedVisibleToolbar: true

    signal openMenu
    signal searchBackward(string searchString)
    signal searchForward(string searchString)
    signal stopSearch

    onSearchModeChanged: if (!searchMode) opened = true

    width: parent.width
    height: content.height
    y: searchMode || opened ? 0 : -height

    Behavior on y { NumberAnimation { duration: 200 } }


    Loader {
        id: content

        width: parent.width
        sourceComponent: searchMode ? itemSearch : itemHeader
    }

    Rectangle {
        id: background

        anchors.fill: parent
        z: -1
        color: Qt.tint(palette.colorScheme === Theme.LightOnDark ? "black" : "white",
                       Theme.rgba(palette.primaryColor, Theme.opacityFaint))
    }

    Component {
        id: itemHeader

        Item {
            height: pageHeader.height

            PageHeader {
                id: pageHeader

                width: parent.width
                title: document.fileName
                titleColor: Theme.highlightColor
                rightMargin: iconMenu.width
                leftMargin: Theme.horizontalPageMargin + Theme.paddingSmall
            }

            IconButton {
                id: iconMenu

                onClicked: openMenu()

                anchors.verticalCenter: parent.verticalCenter
                anchors.right: parent.right
                icon.source: "image://theme/icon-m-menu"
                opacity: root.enabled ? 1.0 : Theme.opacityHigh

                Behavior on opacity { FadeAnimator {}}
            }            
            Component.onCompleted: root.allowedVisibleToolbar = true
        }
    }

    Component {
        id: itemSearch

        Item {
            height: searchField.height

            SearchField {
                id: searchField

                property bool allowedVisibleToolbar: (orientation & Orientation.PortraitMask) || !activeFocus

                width: parent.width
                canHide: false
                placeholderText: qsTrId("docviewer-la-search_by_doc")
                inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhPreferLowercase | Qt.ImhNoPredictiveText
                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: {
                    searchField.focus = false
                    searchForward(searchField.text)
                }
                leftItem: IconButton {
                    onClicked: {
                        searchMode = false
                        stopSearch()
                    }
                    icon.source: "image://theme/icon-m-cancel"
                }
                rightItem: Item {
                    width: (iconRemove.enabled ? iconRemove.width : 0) + (iconUp.enabled ? iconUp.width : 0) + (iconDown.enabled ? iconDown.width : 0)
                    height: iconRemove.height

                    IconButton {
                        id: iconUp

                        onClicked: searchBackward(searchField.text)

                        anchors.left: parent.left
                        icon.source: "image://theme/icon-m-up"
                        enabled: !searchField.focus && searchField.text.length > 0
                        opacity: enabled ? 1.0 : 0.0
                        Behavior on opacity { FadeAnimation {} }
                    }

                    IconButton {
                        id: iconDown

                        onClicked: searchForward(searchField.text)

                        anchors.left: iconUp.right
                        icon.source: "image://theme/icon-m-down"
                        enabled: !searchField.focus && searchField.text.length > 0
                        opacity: enabled ? 1.0 : 0.0
                        Behavior on opacity { FadeAnimation {} }

                    }

                    IconButton {
                        id: iconRemove

                        onClicked: {
                            if (searchField.canHide && searchField.text.length === 0) {
                                searchField.hideClicked()
                            } else {
                                searchField.text = ""
                            }
                        }

                        anchors.left: parent.left
                        icon.source: searchField.canHide && searchField.text.length === 0
                                     ? "image://theme/icon-m-input-remove"
                                     : "image://theme/icon-m-clear"

                        enabled: searchField.focus && searchField.text.length > 0

                        opacity: enabled ? 1.0 : 0.0
                        Behavior on opacity { FadeAnimation {} }
                    }
                }

                onHideClicked: searchMode = false
                onAllowedVisibleToolbarChanged: root.allowedVisibleToolbar = allowedVisibleToolbar
                Component.onCompleted: {
                    forceActiveFocus()
                }
            }
        }
    }
}
