/*
 * Copyright (C) 2013-2014 Jolla Ltd.
 * Copyright (C) 2020-2022 Open Mobile Platform LLC.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.docviewer 1.0
import "utils.js" as Utils

Page {
    id: page

    property url source
    property int indexCount
    property string mimeType

    FileInfo {
        id: info
        source: page.source
    }

    SilicaFlickable {
        id: flickable
        anchors.fill: parent
        contentHeight: contentColumn.height + Theme.paddingLarge

        Column {
            id: contentColumn
            width: parent.width

            PageHeader {
                id: detailsHeader
                //: Details page title
                //% "Details"
                title: qsTrId("docviewer-he-details")
            }

            DetailItem {
                //: File path detail of the document
                //% "File path"
                label: qsTrId("docviewer-la-filepath")
                value: info.filePath
                alignment: Qt.AlignLeft
            }

            DetailItem {
                //: File size detail of the document
                //% "Size"
                label: qsTrId("docviewer-la-filesize")
                value: Format.formatFileSize(info.fileSize)
                alignment: Qt.AlignLeft
            }

            DetailItem {
                //: File type detail of the document
                //% "Type"
                label: qsTrId("docviewer-la-filetype")
                value: info.mimeTypeComment
                alignment: Qt.AlignLeft
            }

            DetailItem {
                //: Last modified date of the document
                //% "Last modified"
                label: qsTrId("docviewer-la-lastmodified")
                value: Format.formatDate(info.modifiedDate, Format.DateFull)
                alignment: Qt.AlignLeft
            }

            DetailItem {
                label: {
                    switch (Utils.docviewerType(page.mimeType)) {
                        case "calc":
                            //: Sheet count of the spreadsheet
                            //% "Sheets"
                            return qsTrId("docviewer-la-sheetcount")
                        case "impress":
                            //: Slide count detail of the presentation
                            //% "Slides"
                            return qsTrId("docviewer-la-slidecount")
                        case "writer":
                            //: Page count of the text document
                            //% "Page count"
                            return qsTrId("docviewer-la-pagecount")
                        case "pdf":
                            //: Page count of the text document
                            //% "Page count"
                            return qsTrId("docviewer-la-pagecount")
                        default:
                            //: Index count for unknown document types.
                            //% "Index count"
                            return qsTrId("docviewer-la-indexcount")
                    }
                }
                value: page.indexCount
                alignment: Qt.AlignLeft
            }
        }

        VerticalScrollDecorator {}
    }
}
