/*
 * Copyright (c) 2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#pragma once

#include <QObject>
#include <QSharedDataPointer>
#include <QSharedPointer>
#include <QImage>
#include <QPointF>
#include <QList>
#include <QFuture>

#include "lodocumentprovider.h"

class LoPageProvider : public QObject
{
    Q_OBJECT
public:

    enum RenderFlags {
        NoFlags = 0,
        // Set if annotations are to be rendered.
        Annot = 0x01,
        // Set if using text rendering optimized for LCD display.
        LCDText = 0x02,
        // Grayscale output.
        GrayScale = 0x08,
        // Limit image cache size.
        LimitImageCache = 0x200,
        // Always use halftone for image stretching.
        ForceHalfTone = 0x400,
        // Render for printing.
        Printing = 0x800,
        // Set to disable anti-aliasing on text.
        NoSmoothText = 0x1000,
        // Set to disable anti-aliasing on images.
        NoSmoothing = 0x2000,
        // Set to disable anti-aliasing on paths.
        NoSmoothPath = 0x4000,
        // Set whether to render in a reverse Byte order, this flag is only used when
        // rendering to a bitmap.
        ReverseByteOrder = 0x10
    };

    struct TaskKey
    {
        TaskKey(int pageNumber, const QRect &renderRect, qreal zoom)
            : pageNumber(pageNumber)
            , renderRect(renderRect)
            , zoom(zoom)
        {}
        bool operator==(const TaskKey &other) const { return pageNumber == other.pageNumber && renderRect == other.renderRect && qFuzzyCompare(zoom, other.zoom); }
        int pageNumber;
        QRect renderRect;
        qreal zoom;
    };

    LoPageProvider(int pageIndex, QSharedPointer<LoDocumentProvider> loProvider);
    virtual ~LoPageProvider();

    int pageIndex() const;
    bool isValid() const;

    QFuture<QSizeF> originalSize() const;
    QFuture<QImage> bitmapFull(qreal pageScale, int renderFlags = 0) const;
    QFuture<QImage> bitmapPart(qreal pageScaleX, qreal pageScaleY, int renderFlags = 0, qreal zoom = 1.0, const QPointF &bias = QPointF(), qreal pageScale = 1.0) const;
    QFuture<QList<QObject *>> words();
    void stopBitmapPart(qreal pageScaleX, qreal pageScaleY, qreal zoom = 1.0, const QPointF &bias = QPointF());
    void stopAllBitmapPart();

private slots:
    void onTileRendered(int pageNumber, const QImage &image, const QRect &documentRect, qreal zoom);

private:
    int m_pageIndex;
    QFutureInterface<QList<QObject *>> m_words;
    mutable QFutureInterface<QSizeF> m_originalSize;
    QSharedPointer<LoDocumentProvider> m_loProvider;
    mutable QHash<TaskKey, QFutureInterface<QImage>> m_taskToItems;
    mutable QMutex m_mutex;
};

Q_DECLARE_METATYPE(LoPageProvider::RenderFlags)
Q_DECLARE_METATYPE(QFuture<QList<QObject *>>);

QDebug operator<<(QDebug debug, const LoPageProvider::TaskKey &taskKey);
