/*
 * Copyright (c) 2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QtAlgorithms>
#include <QHash>

#include <debug.h>

#include "lopageprovider.h"
#include "lodocumentprovider.h"

uint qHash(const LoPageProvider::TaskKey &task)
{
    return qHashBits(&task, sizeof(task.pageNumber) + sizeof (task.renderRect)) + qRound(task.zoom * 100);
}

QFuture<QSizeF> LoPageProvider::originalSize() const
{
    if (m_originalSize.isFinished() || m_originalSize.isRunning())
        return m_originalSize.future();

    m_originalSize.reportStarted();
    QSizeF size = QSizeF(m_loProvider->originalSize(m_pageIndex + 1));
    m_originalSize.reportFinished(&size);

    return m_originalSize.future();
}

QFuture<QImage> LoPageProvider::bitmapFull(qreal pageScale, int renderFlags) const
{
    Q_UNUSED(pageScale)
    Q_UNUSED(renderFlags)
    return {};
}

QFuture<QImage> LoPageProvider::bitmapPart(qreal pageScaleX, qreal pageScaleY, int renderFlags, qreal zoom, const QPointF &bias, qreal pageScale) const
{
    Q_UNUSED(renderFlags)
    if (!m_loProvider)
        return {};

    if (!m_originalSize.isFinished())
        return {};

    QFutureInterface<QImage> interface;
    const auto &future = interface.future();

    int width = qRound(originalSize().result().width() * pageScaleX);
    int height = qRound(originalSize().result().height() * pageScaleY);

    QRect tileRect = QRect(bias.x(), bias.y(), width, height);
    auto key = TaskKey(m_pageIndex + 1, tileRect, zoom);

    m_mutex.lock();
    m_taskToItems.insert(key, interface);
    m_mutex.unlock();
    m_loProvider->renderToImage(m_pageIndex + 1, tileRect, tileRect.size() * pageScale, QSize(), zoom);

    interface.reportStarted();

    return future;
}

QFuture<QList<QObject *> > LoPageProvider::words()
{
    return {};
}

void LoPageProvider::stopBitmapPart(qreal pageScaleX, qreal pageScaleY, qreal zoom, const QPointF &bias)
{
    int width = qRound(originalSize().result().width() * pageScaleX);
    int height = qRound(originalSize().result().height() * pageScaleY);
    QRect tileRect = QRect(bias.x(), bias.y(), width, height);
    qCDebug(lcDocviewerRender) << "StopBitmapPart" << tileRect << "page index:" << pageIndex() << "zoom:" << zoom;

    m_mutex.lock();
    if (m_taskToItems.contains({m_pageIndex + 1, tileRect, zoom})) {
        m_loProvider->stopRenderImage(m_pageIndex + 1, tileRect, zoom);
        m_taskToItems.remove({m_pageIndex + 1, tileRect, zoom});
    } else {
        qCDebug(lcDocviewerRender) << "Not stop render tile rect" << tileRect << "page index:" << pageIndex() << "zoom:" << zoom;
    }
    m_mutex.unlock();
}

void LoPageProvider::stopAllBitmapPart()
{
    m_mutex.lock();
    auto keys = m_taskToItems.keys();
    m_mutex.unlock();

    if (keys.count())
        qCDebug(lcDocviewerRender) << "StopAllBitmapPart" << "page index:" << pageIndex() << "tile count:" << keys.count();

    for (auto key : keys) {
        m_loProvider->stopRenderImage(key.pageNumber, key.renderRect, key.zoom);
    }
}

void LoPageProvider::onTileRendered(int pageNumber, const QImage &image, const QRect &documentRect, qreal zoom)
{
     if (pageNumber == m_pageIndex + 1) {
        auto key = TaskKey(pageNumber, documentRect, zoom);

        m_mutex.lock();
        if (m_taskToItems.contains(key)) {
            m_taskToItems[key].reportFinished(&image);
            m_taskToItems.remove(key);
        } else {
            qCWarning(lcDocviewerRender) << "Task render tile not found" << documentRect << "page index:" << m_pageIndex;
        }
        m_mutex.unlock();
    }
}

LoPageProvider::LoPageProvider(int pageIndex, QSharedPointer<LoDocumentProvider> loProvider)
    : m_pageIndex(pageIndex)
    , m_loProvider(loProvider)
{
    connect(m_loProvider.data(), &BaseDocumentProvider::tileRendered, this, &LoPageProvider::onTileRendered, Qt::DirectConnection);
}

LoPageProvider::~LoPageProvider()
{
    stopAllBitmapPart();
}

int LoPageProvider::pageIndex() const
{
    return m_pageIndex;
}

QDebug operator<<(QDebug debug, const LoPageProvider::TaskKey &taskKey)
{
    debug << "[pageNumber=" << taskKey.pageNumber << " zoom=" << taskKey.zoom << " rect=" << taskKey.renderRect << "]";
    return debug;
}
