/*
 * Copyright (c) 2022 - 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#pragma once

#include <QObject>
#include <QSharedPointer>
#include <QHash>
#include <QSet>
#include <QVector>
#include <QPair>

#include <basedocument.h>

class LoDocumentProvider;
class LoDocument : public BaseDocument
{
    Q_OBJECT

public:
    explicit LoDocument(QObject *parent = nullptr);
    ~LoDocument();

    QThreadPool *threadPool() const override;
    QString path() const override;
    QString fileName() const override;
    QSizeF pageSize(int pageNumber) const override;
    int count() const override;
    int error() const override;
    BaseDocumentProvider::DocumentType documentType() const override;
    QList<QString> pageNames() const override;
    QSharedPointer<BasePage> loadPage(int pageIndex) override;
    void startLoadBookmarks() const override;
    QVector<BaseBookmark *> bookmarks() const override;
    int fileVersion() const override;
    bool saveDocumentAs(const QString &path, const QString &format) const override;
    void mousePress(const InternalPosition &internaPos) override;
    void mouseRelease(const InternalPosition &internaPos) override;
    void mouseDoubleClick(const InternalPosition &internaPos) override;
    void startSelectText(const InternalPosition &internaPos) override;
    void endSelectText(const InternalPosition &internaPos) override;
    void resetSelectText() override;
    void sendUnoCommand(const QString &command, const QString &arguments, bool notifyWhenFinished = true) override;
    void getUnoCommandValues(const QString &command) override;
    void startSearch(const QString &string, SearchDirection direction, const QPointF &startPosition) override;
    void stopSearch() override;
    void goToCell(const QString &adressCell) override;
    QList<QRect> currentHighlightedSearch(int pageIndex) const override;
    QList<QRect> allHighlightedSearch(int pageIndex) const override;
    QList<QPair<QString, QRect>> cellSelected(int pageIndex) const override;
    QList<QRect> highlightedTextArea(int pageIndex) const override;
    QString slsectionText() const override;
    QList<QPair<CommentData, QRect>> comments(int pageIndex) const override;
    QPair<Marker, Marker> selectionMarkers() const override;
    QPair<int, QRect> cursor() const override;

public slots:
    void setPath(const QString &path) override;
    // Part (page) for Calc
    void setPartIndex(int partIndex);

private slots:
    void _libreofficeAnswer(int type, const QByteArray &data);
    void _unoAnswer(const QByteArray &command, const QByteArray &result);
    void _textSelectionAnswer(const QString &text);
    void _comments(const QByteArray &data);
    void _searchResult(const QByteArray &data);
    void _searchNotFound(const QByteArray &data);
    void _unoCommandResult(const QByteArray &data);
    void _textSelection(const QByteArray &data);
    void _textSelectionStart(const QByteArray &data);
    void _textSelectionEnd(const QByteArray &data);
    void _invalidateTiles(const QByteArray &data);
    void _invalidateVisibleCursor(const QByteArray &data);
    void _cellCursor(const QByteArray &data);
    void _cellFormula(const QByteArray &data);
    void _cellAdress(const QByteArray &data);
    void _callbackStateChanged(const QByteArray &data);
    int _pageIndexFromTwipY(int y);

private:
    QSharedPointer<LoDocumentProvider> m_loProvider;
    QHash<int, QSizeF> m_pageSizes;
    QHash<int, QSharedPointer<BasePage>> m_loadedPages;
    QSet<int> m_pagesInProcess;
    int m_lastLoadedPage;
    QVector<BaseBookmark *> m_baseBookmarks;
    QPoint m_searchStartPoint;
    QHash<int, QList<QRect>> m_allHighlighted;
    QHash<int, QList<QRect>> m_currentHighlighted;
    QHash<int, QList<QPair<QString, QRect>>> m_cellSelected;
    QHash<int, QList<QRect>> m_textSelected;
    QHash<int, QList<QPair<CommentData, QRect>>> m_comments;
    QString m_currentCellAdress;
    QString m_currentCellFormula;
    QString m_currentCellContent;
    QString m_selectionText;
    QString m_searchString;
    QPair<int, QRect> m_cursor;
    QPair<Marker, Marker> m_selectionMarkers;
    QThreadPool *m_innerThreadPool;
    // Part (page) for Calc
    int m_partIndex;
};

struct CommentData
{
    QString author;
    QString dateTime;
    QString text;
};
