/*
 * Copyright (c) 2021 - 2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#pragma once

#include "documentproviderglobal.h"
#include <atomic>
#include <QFileInfo>
#include <QImage>
#include <QMap>
#include <QMutex>
#include <QObject>
#include <QThreadPool>

class DOCUMENTPROVIDER_EXPORT ImagesCache : public QObject
{
    Q_OBJECT

public:
    struct CacheKey
    {
        CacheKey(const QString &filePath, int pageNumber, const QSize &imageSize, const QRect &rectTile, qreal zoom)
            : fileId(filePath)
            , pageNumber(pageNumber)
            , imageSize(imageSize)
            , rect(rectTile)
            , zoom(zoom)
        {}

        bool operator==(const CacheKey &other) const
        {
            return pageNumber == other.pageNumber && fileId == other.fileId && imageSize == other.imageSize && rect == other.rect
                   && zoom == other.zoom;
        }

        QString fileId;
        int pageNumber;
        QSize imageSize;
        QRect rect;
        qreal zoom;
    };

public:
    ImagesCache(const QString &pathCache, QObject *parent = nullptr);

    ~ImagesCache();

    bool loadImage(const QString &fileId, int pageNumber, const QSize &canvasSize, const QRect &rectTile, qreal zoom);
    void insertImage(const QString &fileId, int pageNumber, const QImage &image, const QRect &rectTile, qreal zoom);

    void loadCache();
    void saveFile(const QString &filePathCache, int pageNumber, const QSize &imageSize, const QRect &rectTile, qreal zoom);
    QString makeCachePath(const QString &fileId, int pageNumber, const QSize &imageSize, const QRect &rectTile, qreal zoom);

signals:
    void imageLoaded(const QString &fileId, int pageNumber, const QImage &image, const QRect &rectTile, qreal zoom);
    void checksumCalculated(const QString &filePath, const QByteArray &checksum);

private:
    QString m_path;
    qint64 m_limit;            // bytes
    qint64 m_currentCacheSize; // bytes
    QHash<quint32, QFileInfo> m_cache;
    QMap<QDateTime, quint32> m_cacheSortedByCreated;
    QHash<CacheKey, QImage> m_writeQueue;
    std::atomic_bool m_loading;
    QThreadPool m_threadPool;
    QMutex m_mutex;
};
