/*
 * Copyright (c) 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#pragma once

#include "documentproviderglobal.h"
#include <atomic>
#include <QFileInfo>
#include <QImage>
#include <QMap>
#include <QMutex>
#include <QObject>
#include <QThreadPool>

class CopyWorker : public QObject, public QRunnable
{
    Q_OBJECT
        public:
    CopyWorker(const QString &pathFrom, const QString &pathTo)
        : pathFrom(pathFrom)
        , pathTo(pathTo)
    {}

signals:
    void ready(bool status);

private:
    void run() override
    {
        QFile::remove(pathTo);
        emit ready(QFile::copy(pathFrom, pathTo));
    }

    QString pathFrom;
    QString pathTo;
};


class DOCUMENTPROVIDER_EXPORT ExportedCache : public QObject
{
    Q_OBJECT

public:
    struct CacheKey
    {
        CacheKey(const QString &fileId, const QString &format)
            : fileId(fileId)
            , format(format)
        {}

        bool operator==(const CacheKey &other) const
        {
            return fileId == other.fileId && format == other.format;
        }

        QString fileId;
        QString format;
    };

public:
    ExportedCache(const QString &pathCache, QObject *parent = nullptr);

    ~ExportedCache();

    bool loadExportedDoc(const QString &fileId, const QString &pathExportedDoc, const QString &format = "pdf");
    void insertExportedDoc(const QString &fileId, const QString &pathExportedDoc, const QString &format = "pdf");

    void loadCache();
    void writeInList(const QString &fileId, const QString &format);
    QString makeCachePath(quint32 hash);

signals:
    void documentExported(const QString &pathExportedDoc, const QString format, bool status);
    void checksumCalculated(const QString &filePath, const QByteArray &checksum);

private:
    QString m_path;
    qint64 m_limit;            // bytes
    qint64 m_currentCacheSize; // bytes
    QHash<quint32, QFileInfo> m_cache;
    QMap<QDateTime, quint32> m_cacheSortedByCreated;
    QHash<CacheKey, QImage> m_writeQueue;
    std::atomic_bool m_loading;
    QThreadPool m_threadPool;
    QMutex m_mutex;
};
