/*
 * Copyright (C) 2020-2022 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "documentprovider.h"

#include <QFileInfo>
#include <QUrl>

BaseDocumentProvider::BaseDocumentProvider(QObject *parent)
    : QObject(parent)
    , m_isLoaded(false)
    , m_isInitialized(false)
    , m_processing(false)
    , m_pageCount(0)
    , m_loadProgress(0)
    , m_error(NoError)
    , m_documentType(Other)
{}

QString BaseDocumentProvider::path() const
{
    return m_path;
}

void BaseDocumentProvider::setPath(const QString &newPath)
{
    if (m_path != newPath) {
        m_path = newPath;
        emit pathChanged();

        QUrl url(m_path);
        setFileName(url.fileName());
        load(url);
    }
}

QString BaseDocumentProvider::fileName() const
{
    return m_fileName;
}

void BaseDocumentProvider::setFileName(const QString &newFileName)
{
    if (m_fileName != newFileName) {
        m_fileName = newFileName;
        emit fileNameChanged();
    }
}

bool BaseDocumentProvider::loaded() const
{
    return m_isLoaded;
}

void BaseDocumentProvider::setLoaded(bool newLoaded)
{
    if (m_isLoaded != newLoaded) {
        m_isLoaded = newLoaded;
        emit loadedChanged();
    }
}

bool BaseDocumentProvider::initialized() const
{
    return m_isInitialized;
}

void BaseDocumentProvider::setInitialized(bool newInitialized)
{
    if (m_isInitialized != newInitialized) {
        m_isInitialized = newInitialized;
        emit initializedChanged();
    }
}

int BaseDocumentProvider::loadProgress() const
{
    return m_loadProgress;
}

void BaseDocumentProvider::setLoadProgress(int newLoadProgress)
{
    if (m_loadProgress != newLoadProgress) {
        m_loadProgress = newLoadProgress;
        emit loadProgressChanged();
    }
}

int BaseDocumentProvider::error() const
{
    return m_error;
}

void BaseDocumentProvider::setError(const int error)
{
    if (m_error == error)
        return;

    m_error = error;
    emit errorChanged(m_error);
}

BaseDocumentProvider::DocumentType BaseDocumentProvider::documentType() const
{
    return m_documentType;
}

void BaseDocumentProvider::setDocumentType(BaseDocumentProvider::DocumentType type)
{
    if (m_documentType != type) {
        m_documentType = type;
        emit documentTypeChanged();
    }
}

bool BaseDocumentProvider::processing() const
{
    return m_processing;
}

void BaseDocumentProvider::setProcessing(bool processing)
{
    if (m_processing == processing)
        return;

    m_processing = processing;
    emit processingChanged(m_processing);
}

QString BaseDocumentProvider::selectedHyperlink() const
{
    return m_selectedHyperlink;
}

void BaseDocumentProvider::setSelectedHyperlink(const QString &selectedHyperlink)
{
    m_selectedHyperlink = selectedHyperlink;
    emit selectedHyperlinkChanged(m_selectedHyperlink);
}

int BaseDocumentProvider::pageCount() const
{
    return m_pageCount;
}

void BaseDocumentProvider::setPageCount(int pageCount)
{
    if (m_pageCount != pageCount) {
        m_pageCount = pageCount;
        emit pageCountChanged();
    }
}

QList<QString> BaseDocumentProvider::pageNames() const
{
    return m_pageNames;
}

void BaseDocumentProvider::setPageNames(const QList<QString> &pageNames)
{
    if (m_pageNames != pageNames) {
        m_pageNames = pageNames;
        emit pageNamesChanged();
    }
}
