/*
 * Copyright (c) 2021 - 2023 Open Mobile Platform LLC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "documentcache.h"
#include "imagescache.h"
#include "exportedcache.h"

#include <QDir>
#include <QFileInfo>
#include <QImage>
#include <QUrl>
#include <QtConcurrent/QtConcurrent>

namespace {
    const auto PATH_CACHE_PAGE = QStandardPaths::writableLocation(QStandardPaths::CacheLocation) + "/images";
    const auto PATH_CACHE_TILE = QStandardPaths::writableLocation(QStandardPaths::CacheLocation) + "/tiles";
    const auto PATH_CACHE_EXPORT = QStandardPaths::writableLocation(QStandardPaths::CacheLocation) + "/export";
} // namespace

class ChecksumWorker : public QRunnable
{
public:
    ChecksumWorker(const QString &filePath, DocumentCache *documentCache)
        : filePath(filePath)
        , documentCache(documentCache)
    {}

private:
    void run() override
    {
        QFile file(filePath);
        if (file.open(QFile::ReadOnly)) {
            QCryptographicHash hash(QCryptographicHash::Md5);

            if (hash.addData(&file))
                emit documentCache->checksumCalculated(filePath, hash.result());
        }
    }

    QString filePath;
    DocumentCache *documentCache;
};

DocumentCache::DocumentCache(QObject *parent)
    : QObject(parent)
    , m_pageCache(new ImagesCache(PATH_CACHE_PAGE, this))
    , m_tileCache(new ImagesCache(PATH_CACHE_TILE, this))
    , m_exportedCache(new ExportedCache(PATH_CACHE_EXPORT, this))
{
    connect(m_pageCache, &ImagesCache::imageLoaded, this, &DocumentCache::imageLoaded);
    connect(m_tileCache, &ImagesCache::imageLoaded, this, &DocumentCache::imageLoaded);
    connect(m_exportedCache, &ExportedCache::documentExported, this, &DocumentCache::documentExported);
}

DocumentCache::~DocumentCache()
{
    QThreadPool::globalInstance()->clear();
    QThreadPool::globalInstance()->waitForDone();
}

bool DocumentCache::loadImage(const QString &fileId, int pageNumber, const QSize &canvasSize, const QRect &rectTile, qreal zoom)
{
#ifdef DEBUG_MODE
    return false;
#endif

    if (rectTile.isNull())
        return m_pageCache->loadImage(fileId, pageNumber, canvasSize, rectTile, zoom);
    else
        return m_tileCache->loadImage(fileId, pageNumber, canvasSize, rectTile, zoom);
}

void DocumentCache::insertImage(const QString &fileId, int pageNumber, const QImage &image, const QRect &rectTile, qreal zoom)
{
    if (rectTile.isNull())
        return m_pageCache->insertImage(fileId, pageNumber, image, rectTile, zoom);
    else
        return m_tileCache->insertImage(fileId, pageNumber, image, rectTile, zoom);
}

bool DocumentCache::loadExportedDoc(const QString &fileId, const QString &pathExportedDoc, const QString &format)
{
    return m_exportedCache->loadExportedDoc(fileId, pathExportedDoc, format);
}

void DocumentCache::insertEportedDoc(const QString &fileId, const QString &pathExportedDoc, const QString &format)
{
    m_exportedCache->insertExportedDoc(fileId, pathExportedDoc, format);
}

DocumentCache *DocumentCache::instance()
{
    static DocumentCache *_instance = nullptr;

    if (_instance == nullptr)
        _instance = new DocumentCache();

    return _instance;
}

void DocumentCache::calculateChecksum(const QString &filePath)
{
    QThreadPool::globalInstance()->start(new ChecksumWorker(filePath, this));
}
