/**
 * SPDX-FileCopyrightText: Copyright 2019 - 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: GPLv2
 */

/*
 * Copyright (C) 2013 - 2014 Jolla Ltd.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2 only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Nemo.Configuration 1.0
import Nemo.DBus 2.0
import Sailfish.Share 1.0
import Aurora.Office.Files 1.0
import Aurora.Office.License 1.0
import ru.omp.docviewer 1.0

Page {
    id: page

    property alias model: filteredModel.sourceModel
    property string title
    property string searchText: searchField.text
    property bool searchEnabled
    property QtObject provider

    property string deletingSource

    property bool _firstActive: true
    property Dialog _licenseDialog

    function deleteSource(source) {
        pageStack.pop()
        deletingSource = source
        var popup = Remorse.popupAction(
                    page,
                    Remorse.deletedText,
                    function() {
                        provider.deleteFile(deletingSource)
                        deletingSource = ""
                    })
        popup.canceled.connect(function() { deletingSource = "" })
    }

    function prepareMimeType(mimeType) {
        if (mimeType === "text/csv") {
            mimeType = "text/plain";
        }
        return mimeType
    }

    allowedOrientations: Orientation.All

    onStatusChanged: {
        if (_firstActive && status == PageStatus.Active && !licenseStatus.isAccepted) {
            _firstActive = false
            _licenseDialog = pageStack.push("../dialogs/AcceptLicenseDialog.qml", PageStackAction.Immediate)
        }
    }

    onSearchEnabledChanged: {
        if (pageStack.currentPage.status === PageStatus.Active) {
            if (searchEnabled) {
                //#bug OS-5091. Calling forceActiveFocus() cause flickering on long tap.
                // Using timer to avoid it.
                activeFocusTimer.running = true
            } else {
                searchField.focus = false
            }
        }
        if (!searchEnabled) searchField.text = ""
    }

    function getSortParameterName(parameter) {
        if (parameter === FilterModel.Name) {
            //% "name"
            return qsTrId("docviewer-me-sort_by_name")
        } else if (parameter === FilterModel.Type) {
            //% "type"
            return qsTrId("docviewer-me-sort_by_type")
        } else if (parameter === FilterModel.Date) {
            //% "date"
            return qsTrId("docviewer-me-sort_by_date")
        }

        return ""
    }

    function openDocument(url, doShow, unsupported) {
        var _documentPage = pageStack.push(docPageComponent,
                                           {
                                             path: url,
                                             trackerDocumentProvider: provider,
                                           },
                                           doShow ? PageStackAction.Immediate : PageStackAction.Animated)

        applicationWindow.docView = _documentPage
                ? _documentPage._docView
                : null
    }

    function pathToUrl(path) {
        if (path.indexOf("file://") === 0) {
            return path
        }

        return "file://" + path.split("/").map(encodeURIComponent).join("/")
    }

    Connections {
        target: applicationWindow

        onSearchRequest: {
            page.searchEnabled = true

            if (!searchField.activeFocus) {
                activeFocusTimer.start()
            }

            applicationWindow.activate()
        }
    }

    FilterModel {
        id: filteredModel
        filterRegExp: RegExp(escapeRegExp(searchText), "i")

        function escapeRegExp(text) {
          return text.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, '\\$&');
        }
    }

    Timer {
        id: activeFocusTimer
        repeat: false
        running: false
        interval: 750

        onTriggered: searchField.forceActiveFocus()
    }

    SilicaListView {
        id: listView

        anchors.fill: parent
        model: filteredModel
        currentIndex: -1 // otherwise currentItem will steal focus

        header: Item {
            width: listView.width
            height: headerContent.height
        }

        Column {
            id: headerContent

            parent: listView.headerItem
            width: parent.width
            height: pageHeader.height + (searchEnabled ? searchField.height : 0)
            Behavior on height {
                NumberAnimation {
                    duration: 150
                    easing.type: Easing.InOutQuad
                }
            }

            PageHeader {
                id: pageHeader
                title: qsTrId("docviewer-ap-name")
            }

            SearchField {
                id: searchField

                width: parent.width
                opacity: page.searchEnabled ? 1.0 : 0.0
                visible: opacity > 0

                //: Document search field placeholder text
                //% "Search documents"
                placeholderText: qsTrId("docviewer-tf-search-documents")

                // We prefer lowercase
                inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhPreferLowercase | Qt.ImhNoPredictiveText
                EnterKey.iconSource: "image://theme/icon-m-enter-close"
                EnterKey.onClicked: focus = false

                Behavior on opacity { FadeAnimation { duration: 150 } }
            }
        }

        Connections {
            target: searchField.activeFocus ? listView : null
            ignoreUnknownSignals: true
            onContentYChanged: {
                if (listView.contentY > (Screen.height / 2)) {
                    searchField.focus = false
                }
            }
        }

        PullDownMenu {
            id: menu

            property bool _searchEnabled

            // avoid changing text state while menu is open
            onActiveChanged: {
                if (active) {
                    _searchEnabled = page.searchEnabled
                }
            }

            MenuItem {
                text: !menu._searchEnabled ? //% "Show search"
                                             qsTrId("docviewer-me-show_search")
                                             //% "Hide search"
                                           : qsTrId("docviewer-me-hide_search")
                onClicked: page.searchEnabled = !page.searchEnabled
            }

            MenuItem {
                //% "Sort by: %1"
                text: qsTrId("docviewer-me-sort_by").arg(getSortParameterName(filteredModel.sortParameter))
                onClicked: {
                    var obj = pageStack.animatorPush("SortTypeSelectionPage.qml")
                    obj.pageCompleted.connect(function(page) {
                        page.sortSelected.connect(function(sortParameter) {
                            filteredModel.sortParameter = sortParameter
                            pageStack.pop()
                        })
                    })
                }
            }
        }

        InfoLabel {
            parent: listView.contentItem
            y: listView.headerItem.y + pageHeader.height + searchField.height + (page.isPortrait ? Theme.itemSizeMedium : Theme.paddingLarge)
            //: View placeholder shown when there are no documents
            //% "No documents"
            text: page.provider.count === 0
                  ? qsTrId("docviewer-la-no_documents")
                  : //% "No documents found"
                    qsTrId("docviewer-la-not-found")
            opacity: (page.provider.ready && page.provider.count === 0) || (searchText.length > 0 && listView.count == 0) ? 1.0 : 0.0
            Behavior on opacity { FadeAnimator {} }
        }

        delegate: ListItem {
            id: listItem

            contentHeight: Theme.itemSizeMedium
            hidden: deletingSource === model.filePath

            Image {
                id: icon
                anchors {
                    left: parent.left
                    leftMargin: Theme.horizontalPageMargin
                    verticalCenter: parent.verticalCenter
                }
                source: Theme.iconForMimeType(model.fileMimeType) + (highlighted ? "?" + Theme.highlightColor : "")
            }
            Column {
                anchors {
                    left: icon.right
                    leftMargin: Theme.paddingMedium
                    right: parent.right
                    rightMargin: Theme.horizontalPageMargin
                    verticalCenter: parent.verticalCenter
                }
                Label {
                    id: label
                    width: parent.width
                    color: listItem.highlighted ? Theme.highlightColor : Theme.primaryColor
                    text: searchText.length > 0 ? Theme.highlightText(model.fileName, searchText, Theme.highlightColor)
                                                : model.fileName
                    textFormat: searchText.length > 0 ? Text.StyledText : Text.PlainText
                    font.pixelSize: Theme.fontSizeMedium
                    truncationMode: TruncationMode.Fade
                }
                Item {
                    width: parent.width
                    height: sizeLabel.height
                    Label {
                        id: sizeLabel
                        text: Format.formatFileSize(model.fileSize)
                        font.pixelSize: Theme.fontSizeExtraSmall
                        color: listItem.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                    }
                    Label {
                        id: dateLabel
                        anchors.right: parent.right
                        text: "%1, %2".arg(Format.formatDate(model.fileRead, Format.TimeValue)).arg(Qt.formatDate(model.fileRead, Qt.DefaultLocaleShortDate))

                        font.pixelSize: Theme.fontSizeExtraSmall
                        color: listItem.highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor

                        Connections {
                            target: timeFormatConfig
                            onValueChanged: dateLabel.text = Qt.binding(function () {return "%1, %2".arg(Format.formatDate(model.fileRead, Format.TimeValue)).arg(Qt.formatDate(model.fileRead, Qt.DefaultLocaleShortDate))})
                        }
                    }
                }
            }

            onClicked: {
                console.log("model.fileDocumentClass " + model.fileDocumentClass + ", model.fileMimeType " + model.fileMimeType)
                if (model.fileDocumentClass === DocumentListModel.UnknownDocument) {
                    console.log("Unknown file format for file " + model.fileName + " with stated mimetype " + model.fileMimeType)
                    openDocument(model.filePath, false, true)
                } else {
                    openDocument(model.filePath, false, false)
                }
            }

            function deleteFile() {
                remorseDelete(function() { page.provider.deleteFile(model.filePath) })
            }

            // TODO: transitions disabled until they don't anymore confuse SilicaListView positioning. JB#33215
            //ListView.onAdd: AddAnimation { target: listItem }
            //ListView.onRemove: RemoveAnimation { target: listItem }

            menu: Component {
                ContextMenu {
                    id: contextMenu
                    MenuItem {
                        //: Share a file
                        //% "Share"
                        text: qsTrId("docviewer-la-share")

                        onClicked: {
                            shareAction.resources = [pathToUrl(model.filePath)]
                            shareAction.trigger()
                        }

                        ShareAction {
                            id: shareAction
                        }
                    }
                    MenuItem {
                        //: Delete a file from the device
                        //% "Delete"
                        text: qsTrId("docviewer-me-delete")
                        onClicked: {
                            listItem.deleteFile()
                        }
                    }
                }
            }
        }

        VerticalScrollDecorator { }
    }

    Component {
        id: docPageComponent

        DocumentPage { }
    }

    LicenseStatus {
        id: licenseStatus
    }

    ConfigurationValue {
        id: timeFormatConfig
        key: "/sailfish/i18n/lc_timeformat24h"
    }

    DBusAdaptor {
        function openFile(url) {
            if (url.length === 0) {
                applicationWindow.activate()
                return
            }

            if (pageStack.depth > 1) {
                if (_licenseDialog === null) {
                    while (pageStack.depth > 1) {
                        pageStack.pop(undefined, PageStackAction.Immediate)
                    }
                    openDocument(url, true, false)
                    applicationWindow.activate()
                } else {
                    _licenseDialog.acceptDestinationAction = PageStackAction.Replace
                    _licenseDialog.acceptDestinationProperties = {
                        path: url,
                        trackerDocumentProvider: provider,
                    }
                    _licenseDialog.acceptDestination = "ru.omp.docviewer.DocumentPage"
                    _licenseDialog.acceptDestinationInstance.statusChanged.connect(_documetPageReturn)
                    applicationWindow.activate()
                }
            } else {
                openDocument(url, true, false)
                applicationWindow.activate()
            }
        }

        function _documetPageReturn() {
            if (_licenseDialog.acceptDestinationInstance.status === PageStatus.Active) {
                _licenseDialog.acceptDestinationInstance.statusChanged.disconnect(_documetPageReturn)
                _licenseDialog = null
            }
        }

        service: "ru.omp.DocViewer"
        path: "/ru/omp/docviewer"
        iface: "ru.omp.DocViewer"
    }
}
