/**
 * SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.4
import Sailfish.Silica 1.0

Item {
    id: root

    readonly property string stateGrid: "grid"
    readonly property string stateList: "list"

    readonly property var fileInfo: fileListModel.getFileInfo(model.filePath)

    width: GridView.view.cellWidth
    height: GridView.view.cellHeight

    state: __documentsView.isGridMode ? root.stateGrid : root.stateList

    states: [
        State {
            name: root.stateGrid

            AnchorChanges {
                target: icon
                anchors {
                    top: parent.top
                    horizontalCenter: parent.horizontalCenter
                }
            }

            PropertyChanges {
                target: icon
                anchors {
                    topMargin: ui.internalPadding
                }
            }

            AnchorChanges {
                target: fileName
                anchors {
                    top: icon.bottom
                    horizontalCenter: parent.horizontalCenter
                }
            }

            PropertyChanges {
                target: fileName
                horizontalAlignment: Text.AlignHCenter
            }
        },
        State {
            name: root.stateList

            AnchorChanges {
                target: icon
                anchors {
                    left: parent.left
                    top: parent.top
                }
            }

            PropertyChanges {
                target: icon
                anchors {
                    leftMargin: ui.internalPadding
                    topMargin: ui.internalPadding
                }
            }

            AnchorChanges {
                target: fileName
                anchors {
                    left: icon.right
                }
            }

            PropertyChanges {
                target: fileName
                anchors {
                    leftMargin: ui.internalPadding
                    verticalCenter: icon.verticalCenter
                }
            }
        }
    ]

    Rectangle {
        id: background

        anchors {
            fill: parent
            bottomMargin: Theme.paddingSmall
            rightMargin: __documentsView.isGridMode ? Theme.paddingSmall : 0
        }

        radius: Theme.dp(6)
        opacity: 0.1
        color: Theme.primaryColor
    }

    Item {
        id: content

        anchors {
            fill: parent
            margins: Theme.paddingSmall
            bottomMargin: Theme.paddingSmall * 2
            rightMargin: __documentsView.isGridMode
                         ? Theme.paddingSmall * 2
                         : Theme.paddingSmall
        }

        Image {
            id: icon

            width: Theme.iconSizeSmallPlus
            height: width

            source: Theme.iconForMimeType(model.fileMimeType)
        }

        Label {
            id: fileName

            readonly property real availableWidth: __documentsView.isGridMode
                                                   ? content.width
                                                   : content.width - Theme.iconSizeMedium

            readonly property string dots: '...'

            readonly property string suffix: '%1%2'.arg(fileName.dots).arg(root.fileInfo.suffix)

            readonly property string elidedText:
                tm.elidedText.substring(0, tm.elidedText.length -
                                        fileName.suffix.length).trim() + fileName.suffix

            anchors {
                right: parent.right
            }

            wrapMode: Label.Wrap
            maximumLineCount: 2
            font.pixelSize: Theme.fontSizeTiny
            color: Theme.primaryColor
            text: tm.isElided
                  ? fileName.elidedText
                  : root.fileInfo.fileName
        }


        TextMetrics {
            id: tm

            readonly property bool isElided: tm.text !== tm.elidedText

            text: root.fileInfo.baseName
            font: fileName.font
            elide: Text.ElideRight
            elideWidth: fileName.availableWidth * 2 - (__documentsView.isGridMode
                                                       ? fm.advanceWidth(fileName.suffix)
                                                       : fm.advanceWidth(fileName.dots))
        }

        FontMetrics {
            id: fm

            font: fileName.font
        }
    }

    QtObject {
        id: ui

        readonly property real calculatedHeight: {
            if (__documentsView.isGridMode) {
                return Theme.paddingSmall + Theme.iconSizeMedium + fileName.height
            }

            return Theme.iconSizeMedium + Theme.paddingSmall * 2
        }

        readonly property real internalPadding:
            (Theme.iconSizeMedium - Theme.iconSizeSmallPlus) / 2
    }
}
