# Написание плагинов

Copyright © 2024–2025 ООО «Открытая мобильная платформа».
Этот документ предоставляется в соответствии
с [Публичной лицензией Creative Commons с указанием авторства версии 4.0 Международная](../../LICENSE.CC-BY-4.0.ru.md).

## Написание пакетов и плагинов

Создание собственных пакетов и плагинов во Flutter позволяет переиспользовать ваши наработки между проектами, а также делиться ими с сообществом разработчиков. Для написания пакета и плагина необходимо выполнить несколько ключевых шагов:

* Во-первых, использование отдельного Git-репозитория для вашего пакета позволит не только управлять его развитием, но легко добавлять его в конечное приложение. Разделение кода пакета и основного проекта упрощает развитие обеих систем.

* Во-вторых, правильное оформление файла pubspec.yaml является необходимым шагом для указания мета-информации о пакете. В этом файле следует указать название пакета, версию, автора и другие важные данные. Файл pubspec.yaml служит основным источником информации для менеджера пакетов `dart pub`, который используется для установки и обновления пакетов. При необходимости также требуется указать и внешние зависимости от других пакетов.

* В-третьих, требуется создать файл  `<название-пакета>.dart` в папке `lib`. В этом файле требуется реализовать основную функциональность пакета. Важно, чтобы API вашего пакета был понятным, а также предоставлял необходимую документацию.

* Наконец, после разработки вашего пакета его можно подключить к конечному приложению через Git-репозиторий. В файле pubspec.yaml основного приложения требуется указать путь к вашему пакету следующим образом:

```
dependencies:
  название_вашего_пакета:
    git:
      url: https://url_вашего_репозитория.git
```

Это позволит внешнему приложению автоматически загрузить и использовать пакет, который вы создали.

Для более глубокого понимания процесса создания пакетов и плагинов во Flutter рекомендуется ознакомиться с официальной документацией по адресу [https://docs.flutter.dev/packages-and-plugins/developing-packages](https://docs.flutter.dev/packages-and-plugins/developing-packages). Она содержит множество полезных примеров и рекомендаций по разработке пакетов, включая тестирование, публикацию и управление версиями.

## Плагины в ОС Аврора

Плагины позволяют взаимодействовать с платформенными API и нативными библиотеками. Существует три основных способа добавить поддержку ОС Аврора для существующих плагинов.

Первый путь — использование готовой реализации уже существующих плагинов от ОМП или сообщества. В качестве примера можно рассмотреть использование плагина path_provider, который пути к папкам системы и пользователя. Чтобы добавить поддержку Авроры, в файле `pubspec.yaml` необходимо указать зависимости следующим образом:

```
dependencies:
  path_provider: ^2.1.5
  path_provider_aurora:
    git:
      url: https://gitlab.com/omprussia/flutter/packages.git
      path: packages/path_provider_aurora
      ref: aurora-path_provider_aurora-0.6.1
```

В примере показано, что требуется указать в зависимостях как основной плагин path_provider (будет скачан с официального репозитория пакетов pub.dev), так и его реализацию для ОС Аврора с помощью ссылки на Git-репозиторий. Такой подход отлично подходит, если для плагина уже есть готовая реализация.

Второй путь — самостоятельно создать Git-репозиторий с платформенной реализацией плагина, или запросить такую реализацию у сообщества. Затем подключить нативный код по  схеме.

Третий путь — самостоятельно реализовать поддержку ОС Аврора в исходных кодах основного проекта. Этот подход требует больше времени и усилий, но дает полный контроль над поведением плагина и позволяет упростить решение конфликтов между зависимостями. В дальнейшем нативная реализация плагина также может быть вынесена в отдельный Git-репозиторий.

## Структура папки aurora

Теперь подробнее разберем платформенный код для ОС Аврора в папке aurora:

- /desktop/*.desktop — файл для запуска приложения через меню ОС Аврора, также в этом файле настраиваются разрешение на доступ к функционалу ОС Аврора, например, доступ к геолокации или документам пользователя.
- папка /flutter — содержит реализацию плагинов Flutter под ОС Аврора, на случай, если в проекте используется платформо-зависимый пакет, у которого по умолчанию нет реализации для ОС Аврора.
- /icons/*.png — иконки приложения, которые будут отображаться в системно меню установленных приложений.
- /rpm/ru.aurora.app_demo.spec — спецификация сборки установочного RPM-пакета для ОС Аврора.
- файл CMakeLists.txt — настройка сборки нативного проекта, здесь прописываются все необходимые для сборки библиотеки. Если потребуется добавить поддержку Qt, то здесь потребуется прописать необходимые библиотеки.
- файл main.cpp — точка входа в нативное приложение для ОС Аврора, которое инициализирует нативные плагины и запускает среду Flutter.

Данные файлы и папки являются ключевыми для корректной установки и запуска приложения в ОС Аврора.

## Механизмы Dart для плагинов

Важно понимать, что эффективное взаимодействие между Flutter и нативными компонентами — это основа создания расширяемых и функциональных приложений.

Начнем с **Platform Channel**. Это основной и наиболее распространенный способ взаимодействия между кодом Flutter и нативными компонентами. Platform Channel позволяет обмениваться сообщениями между средами Dart и нативной (например, для C++ или Java). Принцип работы заключается в использовании асинхронных вызовов или асинхронных событий, что делает его особенно подходящим для операций, которые могут занять некоторое время, таких как работа с файловой системой.

Следующий механизм — это **Client Wrapper**. Этот интерфейс предназначен для взаимодействия между кодом плагина на C++ и библиотекой Flutter Embedder. Он предоставляет механизмы, которые позволяют управлять из нативного кода объектами среды Dart. То есть нативный код (например, на C++) может управлять объектами пользовательского интерфейса и данными, доступными приложению на Flutter.

Третий механизм, который стоит упомянуть, — это **D-Bus**. Данная система межпроцессного взаимодействия доступна в ОС Аврора (и других системах семейства Unix) и позволяет приложениям обмениваться сообщениями и данными друг с другом через общую шину.

Еще один механизм, **Foreign Function Interface (FFI)**, позволяет разработчикам использовать нативные динамические библиотеки (например, на языках программирования C или Rust) напрямую из кода Dart.

Также есть архитектурный шаблон **Federated plugin**, который позволяет проще создавать сложные и большие плагины, разделяя их реализацию на более мелкие плагины: интерфейс плагины, платформенные части. 

## Нативные плагины

Platform Channels — это основной механизм Flutter, позволяющий взаимодействовать между Dart-кодом и нативным кодом.

Обмен двоичными сообщениями по каналам. Каждый канал идентифицируется названием — строкой, например "ru.omprussia/channel1". Код на Dart создает канал с таким именем и получает возможность вызывать в этом канале определенные методы, которые также определяются строкой. А в нативной части реализуется специальный обработчик, который получив такие идентификаторы канала и метода вызывает нативный код и возвращает через буфер результат его работы. Данные передаются в двоичном (binary) формате, что означает, что разработчики должны позаботиться о правильности кодирования и декодирования сообщений. 

Двоичные сообщения в каналах платформы — это сериализованные в JSON объекты, которые затем упаковываются в двоичный вид, что упрощает и ускоряет работу с ними. Подробнее про данный механизм можно изучить в официальной документации [docs.flutter.dev/platform-integration/platform-channels#codec](https://docs.flutter.dev/platform-integration/platform-channels#codec);

## Каналы платформы в разных ОС

На диаграмме показано сравнение общей архитектуры приложения Flutter при работе на различных ОС, а также место каналов платформы в общей структуре.

## Каналы платформы, механизмы

Основные составляющие Platform Channels:

- PluginRegistrar отвечает за регистрацию платформо-зависимых плагинов. Он позволяет системе знать о плагине и подключать его к соответствующим методам и событиям.
- MethodChannel используется для регистрации и взаимодействия с методами платформо-зависимого плагина. Он позволяет Dart-коду вызывать функции на нативной стороне и наоборот, предоставляя двустороннюю связь между Dart и нативным кодом.
- EventChannel служит для регистрации и подписки в Dart на события, отправляемые из нативной части плагина. EventChannel обеспечивает передачу потоков данных от нативной части к приложению Flutter.
- BinaryMessenger — это низкоуровневый компонент, лежащий в основе обмена данными во Flutter Embedder. Он отвечает за передачу сообщений между Dart и нативным кодом.
- EncodableValue используется в обмене данными между компонентами плагина и Flutter. Он занимается кодированием и декодированием данных при их передаче между Dart и нативной частью.
- BasicMessageChannel позволяет обмениваться сообщениями любого типа данных и может эффективно использоваться для текстовых или других неструктурированных сообщений.

Как и для других операционных систем, интеграция с ОС Аврора требует написать реализацию на C++. В этом контексте Platform Channels работают как посредник, который может взаимодействовать с нативными компонентами ОС Аврора через заранее определенные каналы.

Использование Platform Channels предоставляет гибкие механизмы интеграции с платформенной частью.

## Каналы платформы, типы данных

Каналы платформы используют кодирование сообщений, которое поддерживает эффективную двоичную сериализацию простых данных, таких как логические значения, числа, строки, а также списки и карты этих значений. Сериализация и десериализация этих значений в сообщениях происходит автоматически при отправке и получении.

В таблице на слайде показано, как значения Dart принимаются на стороне платформы и наоборот.

## Каналы платформы, Dart

Для того, чтобы использовать каналы платформы требуется создать канал в коде Dart, например в классе `lib/main.dart` можно создать MethodChannel:

```
static const MethodChannel methodChannel =
      MethodChannel('samples.flutter.io/battery');
```

Для вызова нативного метода через созданный канал нужно указать название этого метода:

```
final int? result = await  
      methodChannel.invokeMethod('getBatteryLevel');
```

## Каналы платформы, C++

Для реализации нативной части требуется зарегистрировать плагин в файле `aurora/main.cpp` на C++:

```
PlatformChannelPlugin
   ::RegisterWithRegistrar(aurora::GetPluginRegistrar());
```   

Затем нужно описать интерфейса плагина в файле `aurora/include/platform_channel/platform_channel_plugin.h`:

```
class PLUGIN_EXPORT PlatformChannelPlugin final 
      : public flutter::Plugin { ... }
```

После этого требуется реализовать код плагина и вызвать нужный метод в файле `aurora/platform_channel_plugin.cpp`:

```
if (call.method_name().compare("getBatteryLevel") == 0) 
   { result->Success(onGetBatteryLevel()); }

// Метод OnGetBatteryLevel() должен вернуть EncodableValue, для примера это конкретное значение
EncodableValue PlatformChannelPlugin::onGetBatteryLevel() {
    return 42;
}
```

## Каналы платформы, пример

Специалисты ОМП расширили стандартный пример Flutter по использованию каналов платформы, добавив реализацию для ОС Аврора. 

Официальная документация по работе с каналами платформы во Flutter: [docs.flutter.dev/platform-integration/platform-channels](https://docs.flutter.dev/platform-integration/platform-channels).

Стартовый пример работы с каналами платформы в ОС Аврора:
[gitlab.com/omprussia/flutter/flutter/-/tree/main/examples/platform_channel](https://gitlab.com/omprussia/flutter/flutter/-/tree/main/examples/platform_channel).

## Каналы платформы и Qt

В ОС Аврора можно использовать любые нативные механизмы при реализации платформенного кода Flutter, а значит можно использовать и Qt, который является фреймворком по умолчанию.

Плагин с поддержкой Qt создается аналогично плагину Platform Channels, но обработку сообщений потребуется реализовать с помощью класса Qt, в котором будут использовать классы C++ из Flutter Embedder.

Стоит отменить, что в данном случае Qt используется без QML и какого-либо графического интерфейса, поэтому не получится отображать интерфейс с помощью Qt Widgets или Qt Quick.

Qt является основным инструментом разработки под ОС Аврора и предоставляет готовую реализацию многих необходимых механизмов, поэтому он является предпочитаемым способом реализации плагинов для ОС Аврора.

Пример использования платформенных плагинов совместно с Qt доступен по ссылке:
[gitlab.com/omprussia/flutter/flutter/-/tree/main/examples/platform_channels_qt](https://gitlab.com/omprussia/flutter/flutter/-/tree/main/examples/platform_channels_qt). 

Ссылка на документацию по классам и методам Qt, доступных в ОС Аврора: [developer.auroraos.ru/doc/5.1.3/software_development/reference/public_api](https://developer.auroraos.ru/doc/5.1.3/software_development/reference/public_api).

## Каналы платформы и Qt, пример

Для начала необходимо добавить поддержку нужных модулей Qt в `aurora/CMakeLists.txt`:

```
target_link_libraries(${BINARY_NAME} PUBLIC Qt5::Core Qt5::Network)
```

Затем нужно включить поддержку Qt в `aurora/main.cpp`:

```
aurora::EnableQtCompatibility();
```

После этого можно реализовать класс плагина на основе `QObject` и `flutter::Plugin`:

```
class PLUGIN_EXPORT PlatformChannelsQtPlugin final : public QObject, public flutter::Plugin { .. }
```

Теперь можно настроить объекты `EventChannel` или `MethodChannel`, зарегистрировать плагин.

После этого требуется реализовать нативный функционал и взаимодействие с EventChannel и MethodChannel.

## Каналы платформы и Client Wrapper

Client Wrapper является механизмом взаимодействия c Flutter Embedder, который обеспечивает взаимодействие между нативной частью приложения и логикой, написанной на Dart. Основная идея заключается в том, что код на Dart и на C++ взаимодействуют друг с другом через двоичные интерфейсы (application binary interface, ABI), что ускоряет обмен данными и упрощает взаимодействие. Код на C++ получает возможность управлять объектами и данными в среде Dart.

Основные плюсы Client Wrapper:

- Абстрагирует сложную нативную логику, предоставляя более простые и удобные интерфейсы для использования в приложениях Flutter.
- Позволяет разработчикам использовать унифицированные интерфейсы для работы с разными платформами. Это важно для обеспечения кроссплатформенности и возможности легко адаптировать приложения под разные операционные системы.
- Обеспечивают двустороннюю коммуникацию между Dart и системными API. Благодаря этому, приложения Flutter могут эффективно взаимодействовать с нативным кодом и ОС.
- Предоставляют дополнительный слой безопасности. Разработчики могут добавлять проверки и валидацию данных на стороне клиента перед отправкой их в нативную часть или наоборот.

Использование Client Wrapper дает возможность не только обеспечивать эффективную работу приложений на разных платформах, но и упрощает процесс разработки, обеспечивая гибкость, удобство и расширяемость кода.

## Использование D-Bus

D-Bus (или "Desktop Bus") является популярным механизмом межпроцессного взаимодействия в Linux-системах и доступен в ОС Аврора.

D-Bus — это система обмена сообщениями для приложений, которая позволяет программам взаимодействовать друг с другом. Он обеспечивает высокоуровневый API для межпроцессного взаимодействия.

При создании плагинов можно уйти от написания платформенного кода, если использовать D-Bus. В контексте Flutter для работы с D-Bus создается не традиционный плагин, а пакет. Это обусловлено тем, что взаимодействие идет через утилиту dbus, без написания нативного кода на C++.

D-Bus интегрируется в приложение Flutter через добавление специального пакета dbus. Это позволяет непосредственно из Dart участвовать в подписке и обработке событий D-Bus.

Разработчик должен правильно связать свое приложение с D-Bus, настроив необходимые параметры. Это важно для корректного взаимодействия и обработки D-Bus сообщений.

## Как работает D-Bus

D-Bus, представляет собой службу, запущенную в операционной системе, которая обеспечивает возможность обмена данными между внешними приложениями через общую шину. Эта шина предоставляет централизованное место, где приложения могут отправлять и получать сообщения, что упрощает взаимодействие между ними. Например, если одно приложение требуется отправить данные другому, оно может сделать это через D-Bus, вместо того чтобы разрабатывать сложный механизм прямой связи и обеспечения безопасности этого канала.

Одной из ключевых особенностей D-Bus является наличие фиксированного набора API, который определяет, как внешние приложения могут взаимодействовать с этой шиной. Каждый набор API включает в себя методы и сигналы, которые приложения могут использовать для обмена данными, а также описания форматов данных, что особенно важно для обеспечения совместимости между различными приложениями.

Для подключения к D-Bus приложение должно пройти процесс регистрации. Это включает в себя описание того, какие данные оно будет отправлять и принимать. Процесс регистрации позволяет D-Bus управлять доступом к ресурсам, упрощая взаимодействие между приложениями и гарантируя, что только уполномоченные приложения могут обмениваться информацией.

Безопасность в D-Bus также является важным аспектом. Система поддерживает различные механизмы обеспечения безопасности, чтобы защитить данные и предотвратить несанкционированный доступ.

## D-Bus, события и службы

Для начала работы с D-Bus в сессии пользователя можно выполнить команду в терминале (например, зайдя на устройство через ssh):

```
dbus-send --session --print-reply --dest=org.freedesktop.DBus /org/freedesktop/DBus org.freedesktop.DBus.ListNames
```

Эта команда вернет список доступных сервисов в рамках текущей пользовательской сессии — какие службы доступны для взаимодействия на данный момент. Каждая из этих служб предоставляет определенные возможности и данные, которые могут быть полезны для других приложений. 

Рассмотрим несколько примеров служб, работающих с D-Bus. В первую очередь, у нас есть служба календаря, которая отвечает за хранение и управление событиями и встречами. Следующий пример — служба телефонии, которая интегрируется с различными приложениями для управления голосовыми вызовами и сообщениями. Еще одна важная служба связана с параметрами устройства и предоставляет информацию о статусе зарядки, сетевом соединении и других аспектах работы устройства.

Все эти службы не только обмениваются данными путём вызова методов, но и отправляют сообщения в шину. Когда, например, происходит изменение статуса зарядки устройства или появляется новое событие в календаре, соответствующая служба отправляет сигнал через D-Bus, который могут прослушивать другие приложения. Такие события описаны в документации служб и включают в себя необходимые поля для чтения, что упрощает интеграцию функций в сторонние приложения.

## D-Bus, пример

Шаги по созданию плагина на базе D-Bus:

1. Создать шаблон плагина командой `flutter-aurora create --template=package --org=ru.aurora dbus_demo`, где `ru.aurora` — доменное имя организации в обратном порядке, а `dbus_demo` — название плагина.
2. Добавить пакет dbus с помощью команды `dart pub global activate dbus`.
3. Использовать команду `dart-dbus`, которая генерирует код на Dart и упрощает процесс подключения приложения к системной шине.
4. Доработать код плагина при необходимости, реализовав обращения к нужным методам и данным внешних сервисов.
5. После этого можно использовать созданный пакет в вашем приложении для подписки на события. Все события и сообщения D-Bus обрабатываются непосредственно в Dart-коде. Также потребуется настроить права доступа к D-Bus в файле *.desktop.

Преимущества использования D-Bus:

- Отсутствие нативного кода.
- Упрощение интеграции.

Недостатки:

- Доступно только для ОС Аврора и настольных системах Linux, нет поддержки iOS и Android.

## Плагины FFI

FFI (Foreign Function Interface) во Flutter предоставляет удобный интерфейс для взаимодействия с библиотеками на языке C (интерфейсы на С++ не поддерживаются). Таким образом FFI позволяет использовать существующие нативные библиотеки и API операционной системы.

Работает это следующим образом:

1. С проектом линкуется нативная библиотека.
2. Для заголовочного файла (*.h) этой библиотеки с помощью утилиты `ffigen` генерируются обёртки на Dart. При необходимости код класса-обертки можно написать самостоятельно.
3. В коде Dart используются сгенерированные обёртки.
4. При использовании FFI мы должны самостоятельно обратиться к нативной библиотеке, вызвав необходимый метод.

Преимущества использования FFI:

- Возможность использования проверенных и производительных нативных библиотек.
- Не требует от разработчиков знания платформо-зависимого кода на уровне плагинов.

Недостатки:

- Потенциально увеличенная сложность отладки и тестирования.
- Необходимость в глубоком понимании используемых нативных библиотек и их API.
- Для работы на нескольких платформах нативные библиотеки должны быть идентичными, иначе потребуется реализация плагина под каждую платформу в отдельности.

## Плагины FFI, пример

Для создания плагина с поддержкой ffi требуется сгенерировать шаблон плагина с типом `plugin_ffi`:

```
flutter-aurora create --template=plugin_ffi --platforms aurora --org=ru.aurora ffi_demo
```

Затем требуется добавить в зависимости пакет ffi и активировать его поддержку в ОС Аврора в файле `pubspec.yaml`:

```
dependencies:
  ffi: ^2.0.2
flutter:
  plugin:
    platforms:
      aurora:
        ffiPlugin: true
```

Затем нужно сгенерировать привязку к нативной библиотеке с помощью `ffiegen` и настроек в файле `ffigen.yaml`:

```
flutter-aurora pub run ffigen --config ffigen.yaml
```

После этого можно доработать плагин и подключить к приложению через git-репозиторий.

## Federated plugin

После детального рассмотрения реализации нативных механизмов стоит изучить, как реализовать структуру плагина.
Основной вопрос, который следует решить: «Как правильно добавить поддержку новой платформы?»

Если пойти простым путём — можно добавить поддержку платформы прямо в плагин, добавив отдельную директорию для целевой платформы и реализацию для неё.

Но данный подход имеет недостатки:

- Возникают сложности при разработке и долгосрочной поддержке плагина.
- Повышается риск рассогласования API и поведения плагина.
- Усложняется процесс тестирования и добавления поддержки новых платформ.

Для решения данной проблемы компанией Google был представлен шаблон федеративного плагина — [federated plugin](https://docs.google.com/document/d/1LD7QjmzJZLCopUrFAAE98wOUQpjmguyGTN2wd_89Srs/edit?pli=1&tab=t.0#heading=h.pub7jnop54q0). В его основе лежит пакет [plugin_platform_interface](https://pub.dev/packages/plugin_platform_interface/versions).

Структура federated plugin:

- app-facing package — пакет, с API на Dart, для взаимодействия с platform interface package, в pubspec.yaml подключаем platform interface package и platform packages.
- platform interface package — пакет, который описывает интерфейс плагина, а также вызов платформенной реализации плагина, строится на зависимости [plugin_platform_interface](https://pub.dev/packages/plugin_platform_interface/versions).
- platform packages (Android Platform Package, iOS Platform Package, Aurora Platrform Package) — пакеты, реализующие платформенную реализацию platform interface package.

## Кодогенераторы

Кодогенераторы — утилиты, которые на базе входных настроек создают заготовки исходных кодов. Для Flutter есть кодогенераторы плагинов, которые ускоряют разработку. Обычно кодогенераторы распространяются в двоичном виде для запуска из консоли. Нужные параметры передаются через входную строку, а данные и расширенные настройки содержатся в конфигурационных файлах.

## Кодогенераторы, ffigen

`ffigen` предназначен для генерации оберток Dart при работе с нативными библиотеками, написанными на языке C. Основным источником для генерации является заголовочный файл на языке C, который имеет расширение *.h. Именно в этих файлах содержится определение функций, которые будут использоваться в приложении на Flutter.

Для использования `ffigen` его необходимо добавить в зависимости проекта. Для этого в файле `pubspec.yaml` Flutter-приложения требуется добавить `ffigen` в секцию `dev_dependencies`. Это обеспечит правильную интеграцию кодогенератора в нашу сборку. После этого потребуется обновить зависимости, выполнив команду `flutter pub get` в терминале.

Следующий шаг – это линковка вашего Flutter-приложения с необходимой нативной библиотекой. Для ОС Аврора это нужно сделать с помощью настройки файла `aurora/CMakeLists.txt`.

После этого можно перейти к настройке самого `ffigen` в файле `pubspec.yaml`. В секцию `ffigen` нужно добавить следующие параметры для генерации обертки `example_lib_bindings.dart` для нативного заголовочного файла `example_lib.h`:

```
ffigen:
  output: 'example_lib_bindings.dart'
  headers:
    entry-points:
      — 'example_lib.h'
```

После выполнения данных шагов, ffigen сгенерирует Dart-файл, который можно будет использовать в коде приложения. 

Для более подробного изучения ffigen можно ознакомиться с примерами и документацией, доступными на сайте pub.dev по ссылке: [pub.dev/packages/ffigen](https://pub.dev/packages/ffigen).

Также, для практического освоения ffigen можно использовать пошаговая инструкция на Google Codelabs, доступную по адресу: [codelabs.developers.google.com/codelabs/flutter-ffigen](https://codelabs.developers.google.com/codelabs/flutter-ffigen).

## Кодогенераторы, pigeon

[Pigeon](https://pub.dev/packages/pigeon) — это пакет Flutter, который генерирует шаблон плагина на базе каналов платформы, используя шаблонное описание конечного API. Pigeon помогает автоматизировать кодогенерацию для Platform Channels, что улучшает читаемость и поддержку проекта.

Преимущества использования Pigeon:

- Pigeon генерирует код, который позволяет убедиться, что данные передаются между Dart и нативным кодом безопасно и с соблюдением нужных типов.
- Pigeon автоматически обновляет нативные заголовочные файлы, если изменяются интерфейсы, что минимизирует ошибки ручного кодирования.
- Pigeon генерирует шаблоны вызовов методов и обработчики в платформенном коде, позволяя разработчикам сосредоточиться на логике приложения, а не на деталях передачи данных.

Одной из особенностей Pigeon является то, что на вход он принимает заранее заданные структуры данных и базовый интерфейс на языке Dart. В этом интерфейсе можно определить необходимые параметры и методы, которые будут использоваться в плагине. Затем Pigeon обрабатывает эти данные и генерирует нужный код на Dart, а также для всех поддерживаемых платформ, заметно сокращая время на реализацию. Сразу стоит добавить, что официально ОС Аврора не поддерживается в pigeon, однако на ОС Аврора будет работать сгенерированный код для Linux (C++ на базе GObject) и Windows (чистый C++).

Кроме того, Pigeon берет на себя конвертацию данных между Dart и нативной частью, написанной на C++. Это означает, что вы можете работать с привычными для вас структурами данных на Dart, а инструмент автоматически позаботится о преобразованиях, необходимым для вызова нативного кода. Такой подход не только экономит время, но и уменьшает вероятность ошибок, связанных с ручным управлением данными.

Как работает Pigeon? Вы создаете файл Dart, в котором определяете интерфейсы данных и методы, которые должны использоваться для общения между Dart и нативными кодами. Затем с помощью команды flutter pub run pigeon генерируется код Dart и нативный, соответствующий описанным интерфейсам. Сгененированный код уже можно использовать в Dart, но нативная часть будет выдавать ошибку, поэтому ее тоже потребуется реализовать.

Ознакомиться с Pigeon и его возможностями можно по следующему адресу: [pub.dev/packages/pigeon](https://pub.dev/packages/pigeon). 

Использование кодогенераторов плагинов, таких как pigeon и ffigen, значительно упрощает задачи интеграции между Dart и нативными библиотеками.

## Pigeon, пример, шаблон и настройки

Для начала требуется добавить пакет pigeon в проект, секция `dev_dependencies` файла `pubspec.yaml`:

```
dev_dependencies:
  pigeon: ^24.1.1
```

Создать файл на Dart с параметрами плагина `<название_плагина>.dart` (далее в примере будет использовать `message` в качестве названия плагина):

```
@ConfigurePigeon(PigeonOptions(
  dartOut: 'lib/<название_плагина>.g.dart',
  dartOptions: DartOptions(),
  cppOptions: CppOptions(namespace: '<название_плагина>'),
  cppHeaderOut: 'aurora/flutter/<название_плагина>.g.h',
  cppSourceOut: 'aurora/flutter/<название_плагина>.g.cpp',
  // настройки под другие платформы
))
// Простой пример плагина для двустороннего обмена данными через Message
class Message { String? сontent; }

@HostApi()
abstract class MessageApi {
  Message sendMessage(Message message);
}
```

## Pigeon, пример, подключение

Далее идет важный шаг — сгенерировать обертки подо все платформы:

```
dart-aurora run pigeon --input <название_плагина>.dart
```

После этого можно добавить сгененированный файл cpp в `aurora/CMakeLists.txt`:

```
add_executable(${BINARY_NAME} main.cpp ${FLUTTER_DIR}/generated_plugin_registrant.cpp ${FLUTTER_DIR}/message.g.cpp)
```

Теперь все обертки настроены и можно реализовать нативную часть плагина (в нашем случае — просто вернутся строка "hello world"):

```
#include "flutter/message.g.h"
using namespace pigeon_demo;
class AuroraMessageApi final: public MessageApi {
public:
    ErrorOr<Message> SendMessage(const Message& message) override {
        std::string content = "hello world";
        Message response(&content);
        return ErrorOr<Message>(response);
    }
};
```

## Pigeon, пример, использование

Для того, чтобы нативную часть можно было использовать, ее требуется зарегистрировать в файле `aurora/main.cpp` (Message — название нашего плагина):

```
auto api = new AuroraMessageApi();
MessageApi::SetUp(aurora::GetPluginRegistrar()->messenger(), api);
```

После этого в коде Dart можно подключить сгенерированный класс:

```
import 'message.g.dart';

Использовать плагин:
final Message message = Message(
  content: 'demo text',
);
final MessageApi api = MessageApi();
final Message result = await api.sendMessage(message);
```

После этого в `result` будет содержаться результат работы плагина. При обновлении файла-шаблона `<название_плагина>.dart` потребуется перегенирировать обертки и обновить нативную часть.

## Сравнение типов плагинов

В таблице представлено сравнение вариантов реализации платформо-зависимых плагинов для ОС Аврора. Однако стоит учитывать, что при реализации нативной части существующего плагина для ОС Аврора придется использовать тот подход, который уже выбрал создатель плагина.

## Где спросить

По вопросам разработки приложений и плагинов Flutter для ОС Аврора можно обращаться к профессионалам в сообществе разработчиков: 
[t.me/aurora_devs](https://t.me/aurora_devs).
