#include <math.h>
#include <qmath.h>
#include <qendian.h>

#include <QDebug>

#include "audioinput.h"

CaptureLevel::CaptureLevel(const QAudioFormat &format)
    : m_level(0)
    , m_maxRms(1)
{
    Q_ASSERT(format.sampleSize() / 8 == 2);
    Q_ASSERT(format.channelCount() == 1);
    Q_ASSERT(format.byteOrder() == QAudioFormat::LittleEndian);
}

void CaptureLevel::start(QIODevice *device)
{
    m_device = device;
    connect(device, &QIODevice::readyRead,
            this, &CaptureLevel::onReadyRead);
    connect(device, &QIODevice::aboutToClose,
            this, &CaptureLevel::stop);
}

void CaptureLevel::stop()
{
    disconnect(m_device, &QIODevice::readyRead,
            this, &CaptureLevel::onReadyRead);
}

void CaptureLevel::onReadyRead()
{
    qint64 acc = 0;
    QByteArray data = m_device->readAll();
    unsigned int nSamples = data.size() / 2;
    if (nSamples) {
        uchar *ptr = reinterpret_cast<uchar *>(data.data());
        for (unsigned int i = 0; i < nSamples; i++) {
            qint16 value = qFromLittleEndian<qint16>(ptr);
            acc += (qint64)value * value;
            ptr += sizeof(qint16);
        }

        qint32 rms = qSqrt(acc / nSamples);
        if (m_maxRms < rms)
            m_maxRms = rms;

        float level = (float)rms / m_maxRms;
        if (fabs(level - m_level) > 0.1) {
            m_level = level;
            emit levelChanged(level);
        }
    }
}

float CaptureLevel::level() const
{
    return m_level;
}

AudioInput::AudioInput()
    : QObject()
{
}

void AudioInput::start()
{
    if (!m_audioInput)
        initializeAudio(QAudioDeviceInfo::defaultInputDevice());
}

void AudioInput::stop()
{
    m_audioInput.reset();
    m_capture.reset();
}

void AudioInput::initializeAudio(const QAudioDeviceInfo &deviceInfo)
{
    QAudioFormat format;
    format.setSampleRate(44100);
    format.setChannelCount(1);
    format.setSampleSize(16);
    format.setCodec("audio/pcm");
    format.setByteOrder(QAudioFormat::LittleEndian);
    format.setSampleType(QAudioFormat::SignedInt);

    if (!deviceInfo.isFormatSupported(format)) {
        qWarning() << "Default format not supported - trying to use nearest";
        format = deviceInfo.nearestFormat(format);
    }

    m_capture.reset(new CaptureLevel(format));
    QObject::connect(m_capture.data(), &CaptureLevel::levelChanged,
                     this, &AudioInput::rmsChanged);
    m_audioInput.reset(new QAudioInput(deviceInfo, format));

    m_capture->start(m_audioInput->start());
}

float AudioInput::rms() const
{
    return m_capture ? m_capture->level() : 0.0;
}
