/**
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <MGConfItem>

#include <auroraapp.h>
#include <callfactoryinterface.h>

#include "audiocallmanager.h"
#include "audiocall.h"

using ru::auroraos::call::AudioContext;
using ru::auroraos::call::Call;
using ru::auroraos::call::CallManager;
using ru::auroraos::call::CallFactoryInterface;

static const auto SCHEME = QStringLiteral("ruauroraosaudiocallexample.://");

class AudioCallFactory : public CallFactoryInterface
{
    Call *newCall(CallManager *manager,
                 const QString &id,
                 const QVariantMap &parameters,
                 QObject *parent) override {
        return new AudioCall(manager, id, parameters, parent);
    }
};

Q_GLOBAL_STATIC(AudioCallFactory, audioCallFactory);

AudioCallManager::AudioCallManager(const QString &accountId, QObject *parent)
    : CallManager(accountId,
        MGConfItem(USE_CALL_ICON_CONFIG_PATH).value().value<bool>()
            ? Aurora::Application::pathTo("svgs/icon-l-checksums.svg").toLocalFile()
            : "",
        parent)
    , m_audioContext(this)
    , m_callId(1)
    , m_holdable(true)
    , m_redialable(true)
{
    setCallFactory(audioCallFactory);
}

Call *AudioCallManager::newCall(QVariantMap &properties)
{
    // Do not allow calls with empty remoteHandle (who do we call?).
    QString remoteHandle = properties[QStringLiteral("remoteHandle")].value<QString>();
    if (remoteHandle.isEmpty())
        return nullptr;

    // Display remoteHandle if a human-readable remoteName is not provided.
    QString remoteName = properties[QStringLiteral("remoteName")].value<QString>();
    if (remoteName.isEmpty())
        properties[QStringLiteral("remoteName")] = remoteHandle;

    // Whether SetHold() calls are welcome.
    properties[QStringLiteral("holdable")] = m_holdable;

    // Fill local account name.
    properties[QStringLiteral("localHandle")] = QStringLiteral("CallExample");
    properties[QStringLiteral("localName")] = QStringLiteral("CallExample");

    if (m_redialable)
        properties[QStringLiteral("uri")] = SCHEME + remoteHandle;

    QString id = QString::number(m_callId++);

    // Add audio control interface.
    auto call = CallManager::newCall(id, properties);
    call->addInterfaces({ audioManager() });
    return call;
}

Call *AudioCallManager::initiateCall(const QVariantMap &props)
{
    QVariantMap properties(props);
    properties[QStringLiteral("incoming")] = false;

    return newCall(properties);
}

Call *AudioCallManager::reportIncomingCall(const QVariantMap &props)
{
    QVariantMap properties(props);
    properties[QStringLiteral("incoming")] = true;

    return newCall(properties);
}

void AudioCallManager::disconnectAll()
{
    for (const auto &call : calls()) {
        removeCall(call);
    }
}

void AudioCallManager::openUri(const QString &uri)
{
    qInfo() << uri;

    if (uri.startsWith(SCHEME)) {
        QString tmp(uri);
        const QString handle = tmp.replace(SCHEME, QStringLiteral(""));
        if (!handle.isEmpty()) {
            for (const auto &call : calls()) {
                if (uri == getCall(call)->uri()) {
                    // The call is already set up. Show the corresponding UI page.
                    qInfo() << "display the current call" << getCall(call)->callId();
                    return;
                }
            }

            QVariantMap properties;
            properties[QStringLiteral("incoming")] = false;
            properties[QStringLiteral("remoteHandle")] = handle;
            // FIXME: Parse remoteName from URI.
            properties[QStringLiteral("remoteName")] = QStringLiteral("Outgoing call ") + handle;

            newCall(properties);
            return;
        }
    }
    qWarning() << "Invalid URI" << uri;
}

bool AudioCallManager::holdable() const
{
    return m_holdable;
}

void AudioCallManager::setHoldable(bool on)
{
    if (on != m_holdable) {
        m_holdable = on;
        emit holdableChanged(on);
    }
}

bool AudioCallManager::redialable() const
{
    return m_redialable;
}

void AudioCallManager::setRedialable(bool on)
{
    if (m_redialable != on) {
        m_redialable = on;
        emit redialableChanged(on);
    }
}

QString AudioCallManager::audioGeneratorConfigPath() const
{
    return QString(AUDIO_GENERATOR_CONFIG_PATH);
}

QString AudioCallManager::useCallIconConfigPath() const
{
    return QString(USE_CALL_ICON_CONFIG_PATH);
}

AudioCall *AudioCallManager::getAudioCall(const QString &callId) const
{
    return qobject_cast<AudioCall*>(getCall(callId));
}
