/**
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <QDebug>

#include "audiocall.h"

using ru::auroraos::call::Call;
using ru::auroraos::call::CallManager;

AudioCall::AudioCall(
        CallManager *manager,
        const QString &id,
        const QVariantMap &parameters,
        QObject *parent)
    : Call(manager, id, parameters, parent)
    , m_audioOutputGeneratorConfig(AUDIO_GENERATOR_CONFIG_PATH)
{
    connect(&m_input, &AudioInput::rmsChanged,
            this, &AudioCall::microphoneRmsChanged);
    connect(&m_audioOutputGeneratorConfig,
            &MGConfItem::valueChanged,
            this,
            &AudioCall::audioOutputGeneratorsStatusChanged);

    m_audioOutputGeneratorConfigValue = m_audioOutputGeneratorConfig.value().value<bool>();

#ifdef STANDALONE_DTMF
    auto dtmf = parameters.find("dtmf");
    if (dtmf != parameters.end() && dtmf->value<bool>()) {
        m_supportsDtmf = true;
        addInterfaces({ new StandaloneDtmfHandler(this) });
    } else {
        m_supportsDtmf = false;
    }
#else
    m_supportsDtmf = true;
#endif

    // Start audio streams when the call becomes active.
    connect(this, &Call::statusChanged,
            this, &AudioCall::onCallStatusChanged);

    if (incoming()) {
        connect(this, &Call::accepted, [=]() {
                    qDebug() << "call accepted";
                    setStatus(Call::Active);
                });
    } else {
        connect(&m_answerTimer, &QTimer::timeout, [=]() {
                    qDebug() << "answered";
                    setStatus(Call::Active);
                });
        m_answerTimer.setInterval(3000);
        m_answerTimer.setSingleShot(true);
        m_answerTimer.start();
    }

    connect(this, &Call::rejected, [=]() {
                qDebug() << "call rejected";
            });
    connect(this, &Call::heldChanged, [=](bool held) {
                qDebug() << "held changed" << held;
            });
    connect(this, &Call::disconnected, [=]() {
                qDebug() << "call destroyed";
            });

    onCallStatusChanged(this->status());
}

void AudioCall::SendDTMF(const QString &codes)
{
    qDebug() << codes;
    m_dtmfString += codes;
    emit dtmfStringChanged(m_dtmfString);
}

bool AudioCall::supportsDtmf() const
{
    return m_supportsDtmf;
}

QString AudioCall::dtmfString() const
{
    return m_dtmfString;
}

void AudioCall::accept()
{
    if (incoming())
        setStatus(Call::Active);
}

void AudioCall::disconnect()
{
    setStatus(Call::Disconnected);
}

AudioCall::~AudioCall()
{
    qDebug() << "deleted";
}

float AudioCall::microphoneRms() const
{
    return m_input.rms();
}

void AudioCall::startAudio()
{
    qDebug();
    m_output.start();
    m_input.start();
}

void AudioCall::stopAudio()
{
    qDebug();
    m_output.stop();
    m_input.stop();
}

void AudioCall::onCallStatusChanged(Call::Status)
{
    // Take the actual status.
    Call::Status status = this->status();
    qDebug() << status;

    switch (status) {
    case Call::Dialing:
    case Call::Active:
        if (m_audioOutputGeneratorConfigValue)
            startAudio();
        break;
    default:
        stopAudio();
        break;
    }
}

void AudioCall::audioOutputGeneratorsStatusChanged()
{
    m_audioOutputGeneratorConfigValue = m_audioOutputGeneratorConfig.value().value<bool>();

    if (m_audioOutputGeneratorConfigValue) {
        startAudio();
    } else {
        stopAudio();
    }
}
