/**
 * SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Policy 1.0
import AudioCallExample 1.0
import QtMultimedia 5.6
import QtQuick.Layouts 1.1
import org.nemomobile.configuration 1.0

Page {
    objectName: "mainPage"
    allowedOrientations: Orientation.All

    property int callId: 0
    property bool soundActive
    readonly property bool _showMicrophoneDisableNotice: !AccessPolicy.microphoneEnabled

    readonly property var deviceViewDescription: {
        var d = {}
        d[AudioDevice.Bluetooth]    = { label: "Bluetooth",     icon: "image://theme/icon-m-bluetooth" }
        d[AudioDevice.Earpiece]     = { label: "Earpiece",      icon: "image://theme/icon-m-call" }
        d[AudioDevice.Speaker]      = { label: "Speaker",       icon: "image://theme/icon-m-browser-sound" }
        d[AudioDevice.WiredHeadset] = { label: "Wired Headset", icon: "image://theme/icon-m-headphone" }
        return d
    }

    function nextCallee() {
        var id = callId;
        callId++;
        return id;
    }

    function getAudioDeviceIcon(device) {
        var data = deviceViewDescription[device.type]
        return data ? data.icon : ""
    }

    function getAudioDeviceLabel(device) {
        var data = deviceViewDescription[device.type]
        return device.name ? device.name : (data ? data.label : device.id)
    }

    on_ShowMicrophoneDisableNoticeChanged: _showMicrophoneDisableNotice
                                           ? microphoneDisableNotice.show()
                                           : Notices._dismissCurrent()

    SilicaListView {
        id : list

        anchors {
            fill: parent
            bottomMargin: Theme.paddingMedium
        }

        model: callManager.calls
        header: Column {
            id: content

            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - 2 * Theme.horizontalPageMargin

            PageHeader {
                anchors.horizontalCenter: parent.horizontalCenter
                width: list.width
                title: qsTr("AudioCall")
                extraContent.children: [
                    IconButton {
                        icon.source: "image://theme/icon-m-about"
                        anchors.verticalCenter: parent.verticalCenter

                        onClicked: pageStack.push(Qt.resolvedUrl("AboutPage.qml"))
                    }
                ]
            }

            Label {
                width: parent.width
                text: callManager.error
            }

            Label {
                width: parent.width
                text: "Calls: " + callManager.calls.length
            }

            TextSwitch {
                id: holdableSwitch

                text: "Make holdable calls"
                automaticCheck: false
                checked: callManager.holdable

                onClicked: callManager.holdable = !checked
            }

            TextSwitch {
                id: redialableSwitch
                text: "Can redial"
                automaticCheck: false
                checked: callManager.redialable

                onClicked: callManager.redialable = !checked
            }

            TextSwitch {
                id: dtmfSwitch
                text: "Supports DTMF"
            }

            TextSwitch {
                id: muteSwitch
                text: "Mute microphone"
                automaticCheck: false
                checked: callManager.audioManager.mute
                onClicked: callManager.audioManager.mute = !checked
            }

            TextSwitch {
                text: "Active audio generator"
                automaticCheck: false
                checked: audioOutputGenerator.value
                onClicked: audioOutputGenerator.value = !checked
            }

            TextSwitch {
                text: "Use call icon"
                automaticCheck: false
                checked: useCallIcon.value
                onClicked: useCallIcon.value = !checked
            }

            ComboBox {
                id: audioDevice

                label: "Output audio device"

                automaticSelection: false
                menu: ContextMenu {
                    Repeater {
                        model: callManager.audioManager.devices
                        IconMenuItem {
                            text: getAudioDeviceLabel(modelData)
                            icon.source: getAudioDeviceIcon(modelData)
                            onClicked: callManager.audioManager.outputDeviceIndex = index
                        }
                    }
                }

                currentIndex: callManager.audioManager.outputDeviceIndex
            }

            TextField {
                id: delayValue

                property var _validator: IntValidator{ bottom: 0; top: 10000; }

                label: "Delay before incoming call, ms"
                validator: text.length === 0 ? null : _validator
                placeholderText: "0"
                inputMethodHints: Qt.ImhDigitsOnly

                EnterKey.enabled: text.length > 0
                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: focus = false

                onTextChanged: {
                    if (text.length > 1) {
                        var origText = text
                        while(origText.charAt(0) === '0')
                        {
                            origText = origText.substring(1)
                        }
                        text = origText
                    }
                }
            }

            Timer {
                id: incomingCallTimer
                interval: parseInt(delayValue.text)
                running: false
                repeat: false
                onTriggered: {
                    var callee = nextCallee();
                    var call = callManager.reportIncomingCall({
                                                                  "remoteHandle" : "_" + callee,
                                                                  "remoteName"   : "Incoming call " + callee,
                                                                  "dtmf"         : dtmfSwitch.checked
                                                              });
                }
            }

            GridLayout {
                id: buttonLayout

                readonly property int numberOfColumns: Screen.sizeCategory > Screen.Medium || !isPortrait
                                                       ? 3
                                                       : 2

                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width
                rowSpacing: Theme.paddingMedium
                columnSpacing: rowSpacing
                columns: numberOfColumns

                Button {
                    Layout.fillWidth: true
                    text: "Incoming call"
                    onClicked: incomingCallTimer.restart()
                }

                Button {
                    Layout.fillWidth: true
                    text: "Outgoing call"
                    onClicked: {
                        var callee = nextCallee();
                        var call = callManager.initiateCall({
                                                                "remoteHandle" : "_" + callee,
                                                                "remoteName"   : "Outgoing call " + callee,
                                                                "dtmf"         : dtmfSwitch.checked
                                                            });
                    }
                }

                Button {
                    Layout.fillWidth: true
                    enabled: callManager.calls.length > 0
                    text: "Disconnect all"
                    onClicked: callManager.disconnectAll()
                }
            }

            Item {
                width: parent.width
                height: Theme.paddingMedium
            }
        }

        spacing: Theme.paddingMedium
        delegate: Rectangle {
            anchors{
                leftMargin: -parent.anchors.leftMargin
                rightMargin: -parent.anchors.rightMargin
                horizontalCenter: parent.horizontalCenter
            }

            width: parent.width - 2 * Theme.horizontalPageMargin
            height: callContent.height + Theme.paddingMedium
            color: Theme.rgba(Theme.highlightColor, 0.1)
            radius: Theme.dp(12)

            Column {
                id: callContent

                property var call: callManager.getCall(modelData)

                anchors.horizontalCenter: parent.horizontalCenter
                rightPadding: Theme.horizontalPageMargin
                leftPadding: rightPadding
                topPadding: Theme.paddingMedium
                width: parent.width
                spacing: Theme.paddingMedium

                Label {
                    width: parent.width - 2 * Theme.horizontalPageMargin
                    text: callContent.call.remoteName + " - " + callContent.call.localName
                    elide: Text.ElideRight
                    maximumLineCount: 1
                }

                Row {
                    spacing: Theme.paddingMedium
                    width: parent.width - 2 * Theme.horizontalPageMargin
                    Label {
                        text: "Status: " + callContent.call.status
                    }

                    Icon {
                        source: microphoneRmsLevel.isMuted ? "image://theme/icon-cover-mute"
                                                           : "image://theme/icon-cover-mic"
                        color:  microphoneRmsLevel.isMuted ? Theme.highlightColor
                                                           : Theme.primaryColor
                    }

                    Rectangle {
                        id: microphoneRmsLevel

                        readonly property bool isMuted: callManager.audioManager.mute || _showMicrophoneDisableNotice

                        anchors.bottom: parent.bottom
                        width: 20
                        color: !isMuted ? "green" : "red"
                        height: parent.height * (!isMuted ? callContent.call.microphoneRms : 1)
                    }
                }

                RowLayout {
                    id: callRow

                    readonly property int buttonWidth: callRow.width / 2

                    anchors.horizontalCenter: callContent.horizontalCenter
                    width: parent.width - 2 * Theme.horizontalPageMargin
                    spacing: Theme.paddingMedium

                    Button {
                        text: "Accept"
                        Layout.fillWidth: true
                        enabled: callContent.call.status == Call.Ringing
                        onClicked: callContent.call.accept()
                    }

                    Button {
                        text: "Disconnect"
                        Layout.fillWidth: true
                        onClicked: callContent.call.disconnect()
                    }
                }

                Label {
                    text: callContent.call.dtmfString
                    visible: callContent.call.supportsDtmf
                }
            }
        }

        VerticalScrollDecorator {}
    }

    Notice {
        id: microphoneDisableNotice

        anchor: Notice.Bottom
        verticalOffset: Theme.paddingMedium
        // Since the Notice component doesn't have permanent display, we specify the duration of 1 day
        duration: 3600000 * 24
        text: "A microphone is disabled by using Aurora Device Manager"
    }

    ConfigurationValue {
        id: audioOutputGenerator

        key: callManager.audioGeneratorConfigPath
    }

    ConfigurationValue {
        id: useCallIcon

        key: callManager.useCallIconConfigPath
    }

    Component.onCompleted: if (audioOutputGenerator.value == null) audioOutputGenerator.value = true
}
