/**
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#pragma once

#include <QObject>
#include <QQuickItem>
#include <QVariantMap>
#include <QTimer>
#include <MGConfItem>

#include <audiocontext.h>
#include <call.h>
#include <dtmfinterface.h>

#include "audioinput.h"
#include "audiooutput.h"

#define STANDALONE_DTMF

using ru::auroraos::call::AudioContext;
using ru::auroraos::call::Call;
using ru::auroraos::call::DTMFInterface;

class AudioCall
    : public Call
#ifndef STANDALONE_DTMF
    , public DTMFInterface
#endif
{
    Q_OBJECT
#ifndef STANDALONE_DTMF
    Q_INTERFACES(ru::auroraos::call::DTMFInterface)
#endif
    Q_PROPERTY(bool supportsDtmf READ supportsDtmf CONSTANT)
    Q_PROPERTY(QString dtmfString READ dtmfString NOTIFY dtmfStringChanged)
    Q_PROPERTY(float microphoneRms READ microphoneRms NOTIFY microphoneRmsChanged)

public:
    AudioCall(
        ru::auroraos::call::CallManager *manager,
        const QString &id,
        const QVariantMap &parameters,
        QObject *parent);

    ~AudioCall();
    
    Q_INVOKABLE void accept();
    Q_INVOKABLE void disconnect();

    bool supportsDtmf() const;
    QString dtmfString() const;
    float microphoneRms() const;

signals:
    void dtmfStringChanged(const QString &tones);
    void microphoneRmsChanged(float rms);

private:
    void startAudio();
    void stopAudio();

private slots:
#ifndef STANDALONE_DTMF
    // DTMFInterface
    void SendDTMF(const QString &codes) override;
#else
    friend class StandaloneDtmfHandler;
    void SendDTMF(const QString &codes);
#endif

    void onCallStatusChanged(Call::Status);
    void audioOutputGeneratorsStatusChanged();

private:
    QTimer m_answerTimer;
    AudioInput m_input;
    AudioOutput m_output;
    bool m_supportsDtmf;
    QString m_dtmfString;
    MGConfItem m_audioOutputGeneratorConfig;
    bool m_audioOutputGeneratorConfigValue;
};

QML_DECLARE_TYPE(AudioCall)

#ifdef STANDALONE_DTMF
class StandaloneDtmfHandler : public QObject, public DTMFInterface
{
    Q_OBJECT
    Q_INTERFACES(ru::auroraos::call::DTMFInterface)

public:
    explicit StandaloneDtmfHandler(AudioCall *parent)
        : QObject(parent)
        , m_call(parent)
    {
    }

public slots:
    // DTMFInterface
    void SendDTMF(const QString &codes) override
    {
        m_call->SendDTMF(codes);
    }

private:
    AudioCall *m_call;
};
#endif
